/* GStreamer
 * Copyright (C) <1999> Erik Walthinsen <omega@cse.ogi.edu>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */


#include <gnome.h>
#include <gst/gst.h>

#include <gst/editor/editor.h>
#include <gst/common/gste-debug.h>

/* class functions */
static void gst_editor_pad_class_init (GstEditorPadClass * klass);
static void gst_editor_pad_init (GstEditorPad * pad);
static void gst_editor_pad_realize (GnomeCanvasItem * citem);
static void gst_editor_pad_resize (GstEditorItem * item);
static void gst_editor_pad_repack (GstEditorItem * item);
static gint gst_editor_pad_event (GnomeCanvasItem * citem, GdkEvent * event);
static void gst_editor_pad_repack (GstEditorItem * item);
static void gst_editor_pad_object_changed (GstEditorItem * item,
    GstObject * object);

static void on_pad_parent_unset (GstObject * object, GstObject * parent,
    GstEditorItem * item);

/* utility functions */

static void gst_editor_pad_link_drag (GstEditorPad * pad,
    gdouble wx, gdouble wy);
static void gst_editor_pad_link_start (GstEditorPad * pad);

static void on_pad_status (GtkWidget * unused, GstEditorPadAlways * pad);
static void on_derequest_pad (GtkWidget * unused, GstEditorPadAlways * pad);
static void on_ghost (GtkWidget * unused, GstEditorPadAlways * pad);
static void on_remove_ghost_pad (GtkWidget * unused, GstEditorPadAlways * pad);
static void on_request_pad (GtkWidget * unused, GstEditorPadRequest * pad);
static void on_frobate (GtkWidget * unused, GstEditorPadSometimes * pad);


enum
{
  ARG_0,
};

enum
{
  LAST_SIGNAL
};

static GstEditorItemClass *parent_class;

static GnomeUIInfo always_pad_menu_items[] = {
  GNOMEUIINFO_ITEM_STOCK ("Pad status...", "Query pad caps, formats, etc",
      on_pad_status, "gtk-properties"),
  GNOMEUIINFO_ITEM_STOCK ("Ghost to parent bin...",
      "Ghost this pad to the parent bin",
      on_ghost, "gtk-jump-to"),
  GNOMEUIINFO_SEPARATOR,
  GNOMEUIINFO_END
};

static GnomeUIInfo ghost_pad_menu_items[] = {
  GNOMEUIINFO_ITEM_STOCK ("Pad status...", "Query pad caps, formats, etc",
      on_pad_status, "gtk-properties"),
  GNOMEUIINFO_ITEM_STOCK ("Remove ghost pad",
      "De-request this previously-requested pad",
      on_remove_ghost_pad, "gtk-cancel"),
  GNOMEUIINFO_ITEM_STOCK ("Ghost to parent bin...",
      "Ghost this pad to the parent bin",
      on_ghost, "gtk-jump-to"),
  GNOMEUIINFO_SEPARATOR,
  GNOMEUIINFO_END
};

static GnomeUIInfo requested_pad_menu_items[] = {
  GNOMEUIINFO_ITEM_STOCK ("Pad status...", "Query pad caps, formats, etc",
      on_pad_status, "gtk-properties"),
  GNOMEUIINFO_ITEM_STOCK ("Release request pad",
      "Release this previously-requested pad",
      on_derequest_pad, "gtk-cancel"),
  GNOMEUIINFO_ITEM_STOCK ("Ghost to parent bin...",
      "Ghost this pad to the parent bin",
      on_ghost, "gtk-jump-to"),
  GNOMEUIINFO_SEPARATOR,
  GNOMEUIINFO_END
};

static GnomeUIInfo request_pad_menu_items[] = {
  GNOMEUIINFO_ITEM_STOCK ("Request pad by name...",
      "Request a pad from this template by name",
      on_request_pad, "gtk-select-font"),
  GNOMEUIINFO_SEPARATOR,
  GNOMEUIINFO_END
};

static GnomeUIInfo sometimes_pad_menu_items[] = {
  GNOMEUIINFO_ITEM_STOCK ("Frobate...",
      "Frobate this pad into a cromulate mass of goo",
      on_frobate, "gtk-execute"),
  GNOMEUIINFO_SEPARATOR,
  GNOMEUIINFO_END
};


GType
gst_editor_pad_get_type (void)
{
  static GType pad_type = 0;

  if (!pad_type) {
    static const GTypeInfo pad_info = {
      sizeof (GstEditorPadClass),
      (GBaseInitFunc) NULL,
      (GBaseFinalizeFunc) NULL,
      (GClassInitFunc) NULL,
      NULL,
      NULL,
      sizeof (GstEditorPad),
      0,
      (GInstanceInitFunc) NULL,
    };

    pad_type =
	g_type_register_static (gst_editor_item_get_type (), "GstEditorPad",
	&pad_info, 0);
  }
  return pad_type;
}

GType
gst_editor_pad_always_get_type (void)
{
  static GType pad_always_type = 0;

  if (!pad_always_type) {
    static const GTypeInfo pad_always_info = {
      sizeof (GstEditorPadAlwaysClass),
      (GBaseInitFunc) NULL,
      (GBaseFinalizeFunc) NULL,
      (GClassInitFunc) gst_editor_pad_class_init,
      NULL,
      NULL,
      sizeof (GstEditorPadAlways),
      0,
      (GInstanceInitFunc) gst_editor_pad_init,
    };

    pad_always_type =
	g_type_register_static (gst_editor_pad_get_type (),
	"GstEditorPadAlways", &pad_always_info, 0);
  }
  return pad_always_type;
}

GType
gst_editor_pad_sometimes_get_type (void)
{
  static GType pad_sometimes_type = 0;

  if (!pad_sometimes_type) {
    static const GTypeInfo pad_sometimes_info = {
      sizeof (GstEditorPadSometimesClass),
      (GBaseInitFunc) NULL,
      (GBaseFinalizeFunc) NULL,
      (GClassInitFunc) gst_editor_pad_class_init,
      NULL,
      NULL,
      sizeof (GstEditorPadSometimes),
      0,
      (GInstanceInitFunc) gst_editor_pad_init,
    };

    pad_sometimes_type =
	g_type_register_static (gst_editor_pad_get_type (),
	"GstEditorPadSometimes", &pad_sometimes_info, 0);
  }
  return pad_sometimes_type;
}

GType
gst_editor_pad_request_get_type (void)
{
  static GType pad_request_type = 0;

  if (!pad_request_type) {
    static const GTypeInfo pad_request_info = {
      sizeof (GstEditorPadRequestClass),
      (GBaseInitFunc) NULL,
      (GBaseFinalizeFunc) NULL,
      (GClassInitFunc) gst_editor_pad_class_init,
      NULL,
      NULL,
      sizeof (GstEditorPadRequest),
      0,
      (GInstanceInitFunc) gst_editor_pad_init,
    };

    pad_request_type =
	g_type_register_static (gst_editor_pad_get_type (),
	"GstEditorPadRequest", &pad_request_info, 0);
  }
  return pad_request_type;
}

GType
gst_editor_pad_requested_get_type (void)
{
  static GType pad_requested_type = 0;

  if (!pad_requested_type) {
    static const GTypeInfo pad_requested_info = {
      sizeof (GstEditorPadRequestedClass),
      (GBaseInitFunc) NULL,
      (GBaseFinalizeFunc) NULL,
      (GClassInitFunc) gst_editor_pad_class_init,
      NULL,
      NULL,
      sizeof (GstEditorPadRequested),
      0,
      (GInstanceInitFunc) gst_editor_pad_init,
    };

    pad_requested_type =
	g_type_register_static (gst_editor_pad_get_type (),
	"GstEditorPadRequested", &pad_requested_info, 0);
  }
  return pad_requested_type;
}

GType
gst_editor_pad_ghost_get_type (void)
{
  static GType pad_ghost_type = 0;

  if (!pad_ghost_type) {
    static const GTypeInfo pad_ghost_info = {
      sizeof (GstEditorPadGhostClass),
      (GBaseInitFunc) NULL,
      (GBaseFinalizeFunc) NULL,
      (GClassInitFunc) gst_editor_pad_class_init,
      NULL,
      NULL,
      sizeof (GstEditorPadGhost),
      0,
      (GInstanceInitFunc) gst_editor_pad_init,
    };

    pad_ghost_type =
	g_type_register_static (gst_editor_pad_get_type (), "GstEditorPadGhost",
	&pad_ghost_info, 0);
  }
  return pad_ghost_type;
}

static void
gst_editor_pad_class_init (GstEditorPadClass * klass)
{
  GObjectClass *object_class;
  GnomeCanvasItemClass *citem_class;
  GstEditorItemClass *item_class;

  object_class = G_OBJECT_CLASS (klass);
  citem_class = GNOME_CANVAS_ITEM_CLASS (klass);
  item_class = GST_EDITOR_ITEM_CLASS (klass);

  parent_class = g_type_class_ref (gst_editor_item_get_type ());

  citem_class->realize = gst_editor_pad_realize;
  citem_class->event = gst_editor_pad_event;
  item_class->resize = gst_editor_pad_resize;
  item_class->repack = gst_editor_pad_repack;
  item_class->object_changed = gst_editor_pad_object_changed;

  if (G_TYPE_FROM_CLASS (klass) == GST_TYPE_EDITOR_PAD_ALWAYS) {
    GST_EDITOR_ITEM_CLASS_PREPEND_MENU_ITEMS (item_class, always_pad_menu_items,
	3);
//    item_class->whats_this = gst_editor_pad_always_whats_this;
  } else if (G_TYPE_FROM_CLASS (klass) == GST_TYPE_EDITOR_PAD_SOMETIMES) {
    GST_EDITOR_ITEM_CLASS_PREPEND_MENU_ITEMS (item_class,
	sometimes_pad_menu_items, 2);
//    item_class->whats_this = gst_editor_pad_sometimes_whats_this;
  } else if (G_TYPE_FROM_CLASS (klass) == GST_TYPE_EDITOR_PAD_REQUEST) {
    GST_EDITOR_ITEM_CLASS_PREPEND_MENU_ITEMS (item_class,
	request_pad_menu_items, 2);
//    item_class->whats_this = gst_editor_pad_request_whats_this;
  } else if (G_TYPE_FROM_CLASS (klass) == GST_TYPE_EDITOR_PAD_REQUESTED) {
    GST_EDITOR_ITEM_CLASS_PREPEND_MENU_ITEMS (item_class,
	requested_pad_menu_items, 4);
//    item_class->whats_this = gst_editor_pad_requested_whats_this;
  } else if (G_TYPE_FROM_CLASS (klass) == GST_TYPE_EDITOR_PAD_GHOST) {
    GST_EDITOR_ITEM_CLASS_PREPEND_MENU_ITEMS (item_class, ghost_pad_menu_items,
	4);
//    item_class->whats_this = gst_editor_pad_ghost_whats_this;
  }
}

static void
gst_editor_pad_init (GstEditorPad * pad)
{
  GType type;
  GstEditorItem *item = GST_EDITOR_ITEM (pad);

  type = G_OBJECT_TYPE (pad);

  if (type == GST_TYPE_EDITOR_PAD_ALWAYS) {
    pad->presence = GST_PAD_ALWAYS;
    item->fill_color = 0xffcccc00;
  } else if (type == GST_TYPE_EDITOR_PAD_SOMETIMES) {
    pad->istemplate = TRUE;
    pad->presence = GST_PAD_SOMETIMES;
    item->fill_color = 0xccffcc00;
  } else if (type == GST_TYPE_EDITOR_PAD_REQUEST) {
    pad->istemplate = TRUE;
    pad->presence = GST_PAD_REQUEST;
    item->fill_color = 0xccccff00;
  } else if (type == GST_TYPE_EDITOR_PAD_REQUESTED) {
    pad->presence = GST_PAD_ALWAYS;
    item->fill_color = 0xffcccc00;
  } else if (type == GST_TYPE_EDITOR_PAD_GHOST) {
    pad->presence = GST_PAD_ALWAYS;
    item->fill_color = 0xcccccc00;
  } else {
    g_assert_not_reached ();
  }

  item->outline_color = 0x0;
  EDITOR_DEBUG ("new pad of type %s (%p)\n", g_type_name (G_OBJECT_TYPE (pad)),
      pad);
}

static void
gst_editor_pad_realize (GnomeCanvasItem * citem)
{
  GstEditorItem *item;
  GstEditorPad *pad;

  item = GST_EDITOR_ITEM (citem);
  pad = GST_EDITOR_PAD (citem);

  /* keep in mind item->object can be a Pad or a PadTemplate; they are the same
   * in the mind of a user just wanting to connect two elements. this function
   * should work for both kinds of object. */

  g_return_if_fail (item->object != NULL);

  if (GNOME_CANVAS_ITEM_CLASS (parent_class)->realize)
    GNOME_CANVAS_ITEM_CLASS (parent_class)->realize (citem);

  if (pad->istemplate) {
    pad->issrc =
	(GST_PAD_TEMPLATE_DIRECTION (GST_PAD_TEMPLATE (item->object)) ==
	GST_PAD_SRC);
  } else {
    pad->issrc = (GST_PAD_DIRECTION (GST_PAD (item->object)) == GST_PAD_SRC);
  }

  if (G_OBJECT_TYPE (pad) == gst_editor_pad_ghost_get_type ())
    pad->isghost = TRUE;

  if (pad->issrc || pad->isghost)
    pad->srcbox = gnome_canvas_item_new (GNOME_CANVAS_GROUP (citem),
	gnome_canvas_rect_get_type (),
	"width_units", 1.0,
	"fill_color", "white", "outline_color", "black", NULL);
  if (!pad->issrc || pad->isghost)
    pad->sinkbox = gnome_canvas_item_new (GNOME_CANVAS_GROUP (citem),
	gnome_canvas_rect_get_type (),
	"width_units", 1.0,
	"fill_color", "white", "outline_color", "black", NULL);

  if (!pad->istemplate) {
    GstPad *_pad, *_peer;

    _pad = GST_PAD (item->object);
    _peer = GST_PAD_PEER (_pad);

    if (_peer) {
      /* we have a linked pad */
      GstEditorPad *peer =
	  (GstEditorPad *) gst_editor_item_get ((GstObject *) _peer);

      if (peer) {
	GnomeCanvasItem *link;

	g_message ("linking GUI for %s:%s and %s:%s", GST_DEBUG_PAD_NAME (_pad),
	    GST_DEBUG_PAD_NAME (_peer));
	link = gnome_canvas_item_new (GNOME_CANVAS_GROUP (citem),
	    gst_editor_link_get_type (), NULL);
	if (pad->issrc)
	  gnome_canvas_item_set (link, "src-pad", pad, "sink-pad", peer, NULL);
	else
	  gnome_canvas_item_set (link, "sink-pad", pad, "src-pad", peer, NULL);

	gst_editor_link_link (GST_EDITOR_LINK (link));
      }
    }

    if (GST_IS_REAL_PAD (_pad) && GST_REAL_PAD (_pad)->ghostpads) {
      GstEditorPad *peer;
      GList *l;

      for (l = GST_REAL_PAD (_pad)->ghostpads; l; l = l->next) {
	GnomeCanvasItem *link;

	_peer = GST_PAD (l->data);
	peer = (GstEditorPad *) gst_editor_item_get ((GstObject *) _peer);

	g_return_if_fail (peer != NULL);

	g_message ("linking ghost pad for %s:%s and %s:%s",
	    GST_DEBUG_PAD_NAME (_pad), GST_DEBUG_PAD_NAME (_peer));
	link =
	    gnome_canvas_item_new (GNOME_CANVAS_GROUP (citem),
	    gst_editor_link_get_type (), NULL);
	gnome_canvas_item_set (link, "ghost", TRUE, NULL);
	if (pad->issrc)
	  gnome_canvas_item_set (link, "src-pad", pad, "sink-pad", peer, NULL);
	else
	  gnome_canvas_item_set (link, "sink-pad", pad, "src-pad", peer, NULL);

	gst_editor_link_link (GST_EDITOR_LINK (link));
      }
    }
  }

  if (pad->isghost && !pad->ghostlink) {
    GnomeCanvasItem *link;
    GstPad *_pad, *_peer;
    GstEditorPad *peer;

    _pad = GST_PAD (item->object);
    _peer = (GstPad *) GST_PAD_REALIZE (_pad);

    peer = (GstEditorPad *) gst_editor_item_get ((GstObject *) _peer);

    g_return_if_fail (peer != NULL);

    g_message ("link ghost pad for %s:%s and %s:%s", GST_DEBUG_PAD_NAME (_pad),
	GST_DEBUG_PAD_NAME (_peer));
    link = gnome_canvas_item_new (GNOME_CANVAS_GROUP (citem),
	gst_editor_link_get_type (), NULL);
    gnome_canvas_item_set (link, "ghost", TRUE, NULL);
    if (!peer->issrc)
      gnome_canvas_item_set (link, "src-pad", pad, "sink-pad", peer, NULL);
    else
      gnome_canvas_item_set (link, "sink-pad", pad, "src-pad", peer, NULL);

    gst_editor_link_link (GST_EDITOR_LINK (link));
  }

  item->realized = TRUE;

  /* we know nothing will be derived from us */
  gst_editor_item_resize (item);
}

static void
gst_editor_pad_resize (GstEditorItem * item)
{
  GstEditorPad *pad = GST_EDITOR_PAD (item);

  /* the link box */
  item->t.w += 4.0;
  item->t.h = MAX (item->t.h, 8.0);

  if (pad->isghost)
    item->t.w += 4.0;

  if (!pad->issrc || pad->isghost)
    item->textx = 5.0;

  GST_EDITOR_ITEM_CLASS (parent_class)->resize (item);
}

static void
gst_editor_pad_repack (GstEditorItem * item)
{
  GstEditorPad *pad = GST_EDITOR_PAD (item);

  if (!item->realized)
    return;

  if (pad->srcbox) {
    gnome_canvas_item_set (pad->srcbox,
	"x1", item->width - 2.0,
	"y1", item->height - 2.0, "x2", item->width, "y2", 2.0, NULL);
  }

  if (pad->sinkbox) {
    gnome_canvas_item_set (pad->sinkbox,
	"x1", 0.0, "y1", item->height - 2.0, "x2", 2.0, "y2", 2.0, NULL);
  }

  if (GST_EDITOR_ITEM_CLASS (parent_class)->repack)
    (GST_EDITOR_ITEM_CLASS (parent_class)->repack) (item);
}

static void
gst_editor_pad_object_changed (GstEditorItem * item, GstObject * object)
{
  GstObject *old = item->object;

  if (old)
    g_signal_handlers_disconnect_by_func (old, G_CALLBACK (on_pad_parent_unset),
	item);

  if (object)
    g_signal_connect (object, "parent-unset", G_CALLBACK (on_pad_parent_unset),
	item);

  parent_class->object_changed (item, object);
}

static gint
gst_editor_pad_event (GnomeCanvasItem * citem, GdkEvent * event)
{
  GstEditorPad *pad;
  GstEditorItem *item;
  GstEditorLink *link;

  item = GST_EDITOR_ITEM (citem);
  pad = GST_EDITOR_PAD (citem);

  g_return_val_if_fail (GST_IS_EDITOR_PAD (item), FALSE);

  switch (event->type) {
    case GDK_ENTER_NOTIFY:
      gnome_canvas_item_set (GNOME_CANVAS_ITEM (item->border),
	  "fill_color_rgba", 0xBBDDBB00, NULL);
      break;
    case GDK_LEAVE_NOTIFY:
      gnome_canvas_item_set (GNOME_CANVAS_ITEM (item->border),
	  "fill_color_rgba", item->fill_color, NULL);
      if (pad->unlinking) {
	GstEditorPad *otherpad;

	otherpad =
	    (GstEditorPad *) ((pad ==
		(GstEditorPad *) pad->link->srcpad) ? pad->link->sinkpad : pad->
	    link->srcpad);

	gst_editor_link_unlink (pad->link);
	gst_editor_pad_link_start (otherpad);
      }
      pad->unlinking = FALSE;
      break;
    case GDK_BUTTON_PRESS:
      if (event->button.button == 1) {
	if (!pad->link)
	  gst_editor_pad_link_start (pad);
	else
	  pad->unlinking = TRUE;
	return TRUE;
      }
      break;
    case GDK_BUTTON_RELEASE:
      if (event->button.button == 1) {
	pad->unlinking = FALSE;
	if (pad->linking) {
	  g_assert (pad->link != NULL);

	  gnome_canvas_item_ungrab (citem, event->button.time);
	  link = pad->link;
	  if (!gst_editor_link_link (link)) {
	    /* for some reason, this is segfaulting. let's go with a temporary workaround...
	       g_object_unref (G_OBJECT (bin->link)); */
	    gnome_canvas_item_hide (GNOME_CANVAS_ITEM (link));
	  }
	  pad->linking = FALSE;
	  return TRUE;
	}
      }
      break;
    case GDK_MOTION_NOTIFY:
      if (pad->linking) {
	gdouble x, y;

	x = event->button.x;
	y = event->button.y;
	gst_editor_pad_link_drag (pad, x, y);
	return TRUE;
      }
      break;
    default:
      break;
  }

  if (GNOME_CANVAS_ITEM_CLASS (parent_class)->event)
    return GNOME_CANVAS_ITEM_CLASS (parent_class)->event (citem, event);
  return FALSE;
}

static void
on_pad_parent_unset (GstObject * object, GstObject * parent,
    GstEditorItem * item)
{
  g_object_set (item, "object", NULL, NULL);

  if (GST_EDITOR_PAD (item)->link)
    gst_editor_link_unlink (GST_EDITOR_PAD (item)->link);

  gnome_canvas_item_hide (GNOME_CANVAS_ITEM (item));

  /* we are removed from the element's pad list with the pad_removed signal */
}

static void
gst_editor_pad_link_start (GstEditorPad * pad)
{
  GdkCursor *cursor;

  g_return_if_fail (GST_IS_EDITOR_PAD (pad));
  g_return_if_fail (pad->link == NULL);

  gnome_canvas_item_new (GNOME_CANVAS_GROUP (pad),
      gst_editor_link_get_type (),
      pad->issrc ? "src-pad" : "sink-pad", pad, NULL);

  cursor = gdk_cursor_new (GDK_HAND2);

  gnome_canvas_item_grab (GNOME_CANVAS_ITEM (pad),
      GDK_POINTER_MOTION_MASK | GDK_BUTTON_RELEASE_MASK,
      cursor, GDK_CURRENT_TIME);

  pad->linking = TRUE;
}

static void
gst_editor_pad_link_drag (GstEditorPad * pad, gdouble wx, gdouble wy)
{
  GstEditorItem *item;
  GnomeCanvasItem *underitem, *under = NULL;
  GstEditorPad *destpad = NULL;

  item = GST_EDITOR_ITEM (pad);

  /* if we're on top of an interesting pad */

  if ((underitem =
	  gnome_canvas_get_item_at (GNOME_CANVAS_ITEM (item)->canvas, wx, wy)))
    under = GST_EDITOR_GET_OBJECT (underitem);

  if (under && GST_IS_EDITOR_PAD (under))
    destpad = GST_EDITOR_PAD (under);

  if (destpad && destpad != pad &&
      (!destpad->link || destpad->link == pad->link) &&
      destpad->issrc != pad->issrc)
    gnome_canvas_item_set (GNOME_CANVAS_ITEM (pad->link),
	pad->issrc ? "sink-pad" : "src-pad", destpad, NULL);
  else {
    if (pad->issrc ? pad->link->sinkpad : pad->link->srcpad)
      gnome_canvas_item_set (GNOME_CANVAS_ITEM (pad->link),
	  pad->issrc ? "sink-pad" : "src-pad", NULL, NULL);
    gnome_canvas_item_set (GNOME_CANVAS_ITEM (pad->link),
	"x", wx, "y", wy, NULL);
  }
}

static void
on_pad_status (GtkWidget * unused, GstEditorPadAlways * pad)
{
  g_return_if_fail (GST_IS_EDITOR_PAD_ALWAYS (pad));

  g_print ("pad status\n");
}

static void
on_derequest_pad (GtkWidget * unused, GstEditorPadAlways * pad)
{
  GstPad *rpad = NULL;
  GstElement *element = NULL;

  g_return_if_fail (GST_IS_EDITOR_PAD_REQUESTED (pad));

  g_print ("derequest pad\n");

  rpad = GST_PAD (GST_EDITOR_ITEM (pad)->object);
  element = GST_ELEMENT (GST_OBJECT_PARENT (rpad));

  if (!GST_ELEMENT_CLASS (G_OBJECT_GET_CLASS (element))->release_pad)
    g_warning ("Elements of type %s have not implemented release_request_pad",
	g_type_name (G_OBJECT_TYPE (element)));

  gst_element_release_request_pad (GST_ELEMENT (GST_OBJECT_PARENT (rpad)),
      rpad);
}

static void
on_remove_ghost_pad (GtkWidget * unused, GstEditorPadAlways * pad)
{
  GstPad *rpad = NULL;

  g_return_if_fail (GST_IS_EDITOR_PAD_GHOST (pad));

  g_print ("deghost pad\n");

  rpad = GST_PAD (GST_EDITOR_ITEM (pad)->object);
  gst_element_remove_pad (GST_ELEMENT (GST_OBJECT_PARENT (rpad)), rpad);
}

static void
on_request_pad (GtkWidget * unused, GstEditorPadRequest * pad)
{
  g_return_if_fail (GST_IS_EDITOR_PAD_REQUEST (pad));

  g_print ("request pad\n");
}

static void
on_frobate (GtkWidget * unused, GstEditorPadSometimes * pad)
{
  g_return_if_fail (GST_IS_EDITOR_PAD_SOMETIMES (pad));
}

static void
on_ghost (GtkWidget * unused, GstEditorPadAlways * pad)
{
  GstElement *bin;
  GstPad *p;

  g_return_if_fail (GST_IS_EDITOR_PAD_ALWAYS (pad));

  p = GST_PAD (GST_EDITOR_ITEM (pad)->object);

  bin = (GstElement *) GST_OBJECT_PARENT (GST_OBJECT_PARENT (p));

  gst_element_add_ghost_pad (bin, p, (gchar *) GST_OBJECT_NAME (p));
}
