#include "SocketComm.h"
#include "grubface.h"
#include "input_thread.h"
#include <cstdlib>
#include <cstring>
#include <cstdio>

SocketComm::SocketComm()
{
	constr_init();
}

// server constructor
SocketComm::SocketComm(
	SOCKET server_sock,
	unsigned short port,
	SocketReadFunc read_func,
	SocketErrorFunc error_func )
{
	constr_init();

	this->port = port;
	this->read_func = read_func;
	this->error_func = error_func;
	this->server_sock = server_sock;
}

// client constructor
SocketComm::SocketComm(
	const char *host,
	unsigned short port,
	SocketReadFunc read_func,
	SocketErrorFunc error_func )
{
	constr_init();

	if ( this->host ) {
		
		delete [] this->host;
		this->host = 0;
	}

	this->host = my_strdup( host );
	this->port = port;
	this->read_func = read_func;
	this->error_func = error_func;

	setClient();
}

void SocketComm::constr_init()
{
	is_server = true;
	is_sock_init = false;
	error_msg = 0;
	port = 1;
	read_func = 0;
	error_func = 0;
	error_set = false;
	max_doclen = 1000000000;
	is_threading = false;
	read_func_arg = 0;
	error_func_arg = 0;
#ifndef GRUB_UNIX
	hThread = NULL;
#endif
	sock = INVALID_SOCKET;
	server_sock = INVALID_SOCKET;

	host = my_strdup( "127.0.0.1" );
#ifdef GRUB_UNIX
	pthread_mutex_init(&mutex, (pthread_mutexattr_t *)0 );
#else
	InitializeCriticalSection( &mutex );
#endif
}

SocketComm::~SocketComm()
{
	endSock();

	if ( error_msg ) delete [] error_msg;
	if ( host ) delete [] host;

#ifdef GRUB_UNIX
	pthread_mutex_destroy(&mutex);
#else
	DeleteCriticalSection( &mutex );
#endif
}

void SocketComm::setClient()
{
	is_server = false;
}

void SocketComm::setServerSock( SOCKET sock )
{
	this->sock = sock;
	is_server = true;
	is_sock_init = true;
	gmsg_init( sock );
}

// WATCH: server_sock has nothing to do with setServerSock()
SOCKET SocketComm::getServerSock()
{
	return server_sock;
}

void SocketComm::setHost( const char *host )
{
	if ( this->host ) delete [] this->host;
	this->host = my_strdup(host);
}

void SocketComm::setPort( unsigned short port )
{
	this->port = port;
}

unsigned short SocketComm::getPort()
{
	return port;
}

void SocketComm::setMaxDocLen( unsigned int len )
{
	max_doclen = len;
}

unsigned int SocketComm::getMaxDocLen()
{
	return max_doclen;
}

void SocketComm::setReadCallback( SocketReadFunc read_func )
{
	this->read_func = read_func;
}

SocketReadFunc SocketComm::getReadCallback()
{
	return read_func;
}

void SocketComm::setReadCallbackArg( void *arg )
{
	read_func_arg = arg;
}

void *SocketComm::getReadCallbackArg()
{
	return read_func_arg;
}

void SocketComm::setErrorCallback( SocketErrorFunc error_func )
{
	this->error_func = error_func;
}

void SocketComm::setErrorCallbackArg( void *arg )
{
	error_func_arg = arg;
}

int SocketComm::initSock()
{
	if ( ! is_sock_init ) {

		if ( is_server ) {

			/* server */
			/* startThread() will start a new thread that will
			 * deal with clients connecting, and which will set
			 * the socket */
		}
		else {

			/* client */

			int ret = grub_attach( host, port );
			if ( ret != GF_SUCCESS )
				return -1;

			is_sock_init = true;
		}

		if ( startThread() != 0 )
			return -1;
	}

	return 0;
}

void SocketComm::endSock()
{
	endThread();

	if ( is_server ) {

		if ( sock != INVALID_SOCKET ) {

			closesocket( sock );
			sock = INVALID_SOCKET;
		}
	} else
		grub_detach();  // ret is always GF_SUCCESS

	is_sock_init = false;
}

bool SocketComm::isSockInit()
{
	return is_sock_init;
}

void SocketComm::lock()
{
#ifdef GRUB_UNIX
	pthread_mutex_lock( &mutex );
#else
	EnterCriticalSection( &mutex );
#endif
}

void SocketComm::unlock()
{
#ifdef GRUB_UNIX
	pthread_mutex_unlock( &mutex );
#else
	LeaveCriticalSection( &mutex );
#endif
}

int SocketComm::send( const char *str, unsigned int len )
{
	int fret;

	lock();

	if ( ! is_sock_init || error_set ) {

		unlock();
		return -2;
	}

	/* ret = GF_SUCCESS, GF_NOINIT, GF_SOCKERR, GF_NOTREADY,
	 *       GF_INTRNERR, GF_BADDATA */
	fret = gmsg_send( str, len );
	if ( fret != GF_SUCCESS ) {
		char buf[512];

		sprintf( buf, "send: gmsg_send: %s", gmsg_strerror() );
		unlock();

		// TODO: Make sure there is no race condition
		setError( buf );
		endSock();
		return -1;
	}

	unlock();

	return 0;
}

void SocketComm::setError( const char *msg )
{
	lock();

	if ( ! error_set ) {

		error_set = true;
		thread_die = true;

		if ( error_msg ) {

			delete [] error_msg;
			error_msg = 0;
		}

		if ( msg )
			error_msg = my_strdup( msg );

		unlock();

		if ( error_func ) {

			if ( msg )
				error_func( error_func_arg, msg );
			else
				error_func( error_func_arg, "Unspecified Error occured" );
		}
	}
	else
		unlock();
}

const char *SocketComm::getError()
{
	return error_msg;
}

// private

int SocketComm::startThread()
{
	lock();

	if ( ! is_threading ) {

		thread_die = false;
		is_threading_picked = false;
#ifndef GRUB_UNIX
		hThread = NULL;
#endif
		sock = INVALID_SOCKET;
		if ( createThread() != 0 ) {

			unlock();
			return -1;
		}

		// make sure the thread is running at
		// function exit
		while ( ! is_threading ) {

			unlock();
			msleep( 200 );
			lock();
		}
		is_threading_picked = true;
	}

	unlock();

	return 0;
}

void SocketComm::endThread()
{
	lock();

	// signal thread for his death
	thread_die = true;

	// make sure the thread dies before doing cleanup
	while ( is_threading ) {

		unlock();
		msleep( 200 );
		lock();
	}

	cleanupThread();

	unlock();
}

int SocketComm::createThread()
{
#ifdef GRUB_UNIX
	int ret;

	ret = pthread_create(&thread_id, 0, input_thread_main, this);
	if ( ret != 0 )
		return -1;
#else
	hThread = CreateThread(
		NULL,
		0,
		(unsigned long (__stdcall *)(void *))input_thread_main,
		(void *)this,
		0,
		&thread_id
	);
	if ( hThread == NULL )
		return -1;
#endif
	return 0;
}

void SocketComm::cleanupThread()
{
#ifdef GRUB_UNIX
	pthread_join(thread_id, (void **)0);
#else
	if ( hThread != NULL ) {

		CloseHandle( hThread );
		hThread = NULL;
	}
#endif
}

// like strdup, but using new instead of malloc
char *SocketComm::my_strdup( const char *str )
{
	int len;
	char *new_str;

	len = strlen(str);

	new_str = new char[len + 1];
	strcpy( new_str, str );

	return new_str;
}

void SocketComm::msleep( unsigned int msec )
{
#ifdef GRUB_UNIX
	usleep( 1000 * msec );
#else
	Sleep( (DWORD)msec );
#endif
}
