#include "Msg.h"
#ifdef GRUB_UNIX
#include <sys/types.h>
#include <sys/socket.h>
#endif

// TODOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOO
// undefine this!!!
#define DEBUG_MSG_MANAGER

#ifdef DEBUG_MSG_MANAGER
#include <cstdio>
#endif

#include <cstring>

int Msg_read_callback( void *arg, const char *buf, unsigned int len );
void Msg_error_callback( void *arg, const char *msg );

/////////////////////
// class MsgSender //
/////////////////////

MsgSender::MsgSender() : xml_parser()
{
	reset();
}

MsgSender::~MsgSender()
{
}

void MsgSender::reset()
{
	xml_parser.resetData();
	xml_parser.addSimpleHeader();
	xml_parser.openElement("ConsoleMessage");
}

void MsgSender::setMessageType( const char *type )
{
	xml_parser.addElementString("Type", type);
}

void MsgSender::setMessageName( const char *name )
{
	xml_parser.addElementString("Name", name);
	xml_parser.openElement("Data");
}

void MsgSender::appendDataString( const char *tag_name, const char *value )
{
	xml_parser.addElementString( tag_name, value );
}

void MsgSender::appendDataInteger( const char *tag_name, int value )
{
	xml_parser.addElementInt( tag_name, value );
}

void MsgSender::appendDataDouble( const char *tag_name, double value )
{
	xml_parser.addElementDouble( tag_name, value );
}

const char *MsgSender::getMessage()
{
	xml_parser.closeElement("Data");
	xml_parser.closeElement("ConsoleMessage");

	return xml_parser.getData();
}

///////////////////////
// class MsgReceiver //
///////////////////////

MsgReceiver::MsgReceiver() : xml_parser()
{
}

MsgReceiver::~MsgReceiver()
{
}

void MsgReceiver::reset()
{
	xml_parser.resetData();
}

// ret = 0, -1 (XML parse error)
int MsgReceiver::setMessage( const char *msg )
{
	return xml_parser.parseData( msg );
}

// ret = 1, 0
int MsgReceiver::getMessageType( char *& type )
{
	return xml_parser.parsedValueString(".ConsoleMessage.Type", 0, type );
}

// ret = 1, 0
int MsgReceiver::getMessageName( char *& name )
{
	return xml_parser.parsedValueString(".ConsoleMessage.Name", 0, name );
}

// for all getData..() methods: ret = 1 (found), 0 (not found), -1 (bad format)
int MsgReceiver::getDataString( const char *tag_name, int index, char *& value )
{
	char buf[1024];
	int len;

	strcpy(buf, ".ConsoleMessage.Data.");
	len = strlen(buf);
	strncpy(buf + len, tag_name, 1023 - len );
	buf[1023] = '\0';

	return xml_parser.parsedValueString(buf, 0, value );
}

int MsgReceiver::getDataInteger( const char *tag_name, int index, int& value )
{
	char buf[1024];
	int len;

	strcpy(buf, ".ConsoleMessage.Data.");
	len = strlen(buf);
	strncpy(buf + len, tag_name, 1023 - len );
	buf[1023] = '\0';

	return xml_parser.parsedValueInteger(buf, 0, value );
}

int MsgReceiver::getDataDouble( const char *tag_name, int index, double& value )
{
	char buf[1024];
	int len;

	strcpy(buf, ".ConsoleMessage.Data.");
	len = strlen(buf);
	strncpy(buf + len, tag_name, 1023 - len );
	buf[1023] = '\0';

	return xml_parser.parsedValueDouble(buf, 0, value );
}

///////////////////////
// class MsgReceiver //
///////////////////////

// server constructor
MsgManager::MsgManager( unsigned short portnum ) :
	MsgSender(), MsgReceiver(),
	is_server(true), portnum(portnum), is_active(false), scomm(0),
	srv_sock(INVALID_SOCKET)
{
	this->hostname[0] = '\0';
}

// client constructor
MsgManager::MsgManager( unsigned short portnum, const char *hostname ) :
	MsgSender(), MsgReceiver(),
	is_server(true), portnum(portnum), is_active(false), scomm(0),
	srv_sock(INVALID_SOCKET)
{
	this->hostname[0] = '\0';

	if ( hostname ) {

		// is a client

		is_server = false;
		strncpy(this->hostname, hostname, 1023);
		this->hostname[1023] = '\0';
	}
}

MsgManager::~MsgManager()
{
	deleteSocketComm();

	if ( srv_sock != INVALID_SOCKET )
		closeServer();
}

void MsgManager::setActive( bool active )
{
	if ( active ) {

		is_active = true;
		initSocketComm();
	}
	else {

		is_active = false;
		deleteSocketComm();

		closeServer();
	}
}

bool MsgManager::isActive()
{
	return is_active;
}

bool MsgManager::isServer()
{
	return is_server;
}

unsigned short MsgManager::getPort()
{
	return portnum;
}

const char *MsgManager::getHost()
{
	return hostname;
}

bool MsgManager::isConnected()
{
	bool ret = false;

	if ( is_active && scomm && ! scomm->getError() )
		ret = true;

	return ret;
}

void MsgManager::send()
{
	if ( is_active ) {

		initSocketComm();

		if ( scomm ) {
			int fret;

			// fret = 0, -1, -2
			fret = scomm->send(
				getMessage(),
				strlen(getMessage())
			);
			if ( fret == -1 ) {

				// error sending;
				// do nothing here, as event will be
				// fired and handled in errorCallback()
			}
		}

		// clear the message after it was sent
		MsgSender::reset();
	}
}

void MsgManager::invokeError( const char *msg )
{
	if ( is_active && scomm ) {

		scomm->setError( msg );
	}
}

// private

// on failure srv_sock will be set to INVALID_SOCKET
// will simply exit if server is already set
void MsgManager::establishServer()
{
	char myname[256];
	struct sockaddr_in sa;
	struct hostent *hp;

	if ( srv_sock != INVALID_SOCKET )
		return;

	memset(&sa, 0, sizeof(struct sockaddr_in));
	gethostname(myname, sizeof(myname));
	hp = gethostbyname(myname);
	if (hp == NULL)
		return;

	sa.sin_family = hp->h_addrtype;
	sa.sin_port = htons(portnum);
	srv_sock = socket(AF_INET, SOCK_STREAM, 0);
	if (srv_sock == INVALID_SOCKET)
		return;

	/* bind the socket to the internet address */
	if (bind(srv_sock, (struct sockaddr *)&sa, sizeof(struct sockaddr_in)) ==
		SOCKET_ERROR)
	{
		closesocket(srv_sock);
		srv_sock = INVALID_SOCKET;
		return;
	}
	listen(srv_sock, 3);
}

void MsgManager::closeServer()
{
	if ( srv_sock != INVALID_SOCKET ) {

		closesocket(srv_sock);
		srv_sock = INVALID_SOCKET;
	}
}

// on failure scomm will be set to null
// will simply exit if scomm is already set
void MsgManager::initSocketComm()
{
	if ( scomm ) {

		if ( scomm->getError() ) {

			delete scomm;
			scomm = 0;
		}
		else
			return;
	}

	if ( is_server ) {

		establishServer();

		if ( srv_sock != INVALID_SOCKET ) {

			// server
			scomm = new SocketComm(
				srv_sock,
				portnum,
				Msg_read_callback,
				Msg_error_callback );
			scomm->setReadCallbackArg( (void *)this );
			scomm->setErrorCallbackArg( (void *)this );

			if ( scomm->initSock() != 0 ) {

				delete scomm;
				scomm = 0;
			}
		}
	}
	else {

		// client
		scomm = new SocketComm(
			hostname,
			portnum,
			Msg_read_callback,
			Msg_error_callback );
		scomm->setReadCallbackArg( (void *)this );
		scomm->setErrorCallbackArg( (void *)this );

		if ( scomm->initSock() != 0 ) {

			delete scomm;
			scomm = 0;
		}
	}
}

void MsgManager::deleteSocketComm()
{
	if ( scomm ) {

		delete scomm;
		scomm = 0;
	}
}

// override this method to implement
void MsgManager::readCallback( const char *data )
{
#ifdef DEBUG_MSG_MANAGER
	printf("MsgManager::readCallback: data:\n---BEGIN---\n%s\n---END---\n",
		data );
#endif
}

// override this method and call this one first
void MsgManager::errorCallback( const char *msg )
{
#ifdef DEBUG_MSG_MANAGER
	printf("MsgManager::errorCallback: ERROR: %s\n", msg );
#endif
}

// private functions

int Msg_read_callback( void *arg, const char *buf, unsigned int len )
{
	((MsgManager *)arg)->readCallback( buf );

	return 0;
}

void Msg_error_callback( void *arg, const char *msg )
{
	((MsgManager *)arg)->errorCallback( msg );
}
