/* pmud Monitor panel applet
 *
 * By Uwe Steinmann <uwe@steinmann.cx>
 *
 * Enjoy.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or 
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License  
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#else
#  define PACKAGE "gpmudmon"
#  define VERSION "0.1.0"
#endif
#include <stdio.h>
#include <stdlib.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/time.h>

#include <gnome.h>
#include <applet-widget.h>
#include "gpmudmon.h"
#include "gtkgauge.h"
#include "battery.xpm"
#include "ac.xpm"
#include "charge.xpm"


char **line0_pixmap;
char **line1_pixmap;
GtkWidget *counter;
GtkTooltips *tooltip;

extern void prefs_cb(GtkWidget *w, gpointer p);

int count_timer(gpointer data);
int open_pmudinfo(PmudData *pd);
static PMUD_STATUS * get_pmud_status(PmudData *pd);

static void
about_cb(GtkWidget *w, gpointer d)
{
	gchar *authors[4];
	GtkWidget *about;
	
	authors[0] = "Uwe Steinmann <uwe@steinmann.cx>";
	authors[1] = "http://gehtnix.fernuni-hagen.de/gpmudmon";
	authors[2] = NULL;
	
	about = gnome_about_new (
		"GNOME Pmud Monitor",
		VERSION,
		"(C) 2000 Uwe Steinmann",
		(const gchar**)authors,
		_("This applet displays the battery status as provided by pmud."),
		NULL);
	
	gtk_widget_show(about);
	
	return;
}

static void
sleep_cb(GtkWidget *w, gpointer d)
{
	int error;
	PmudData *pd = (PmudData *) d;
	error = write(pd->pmudinfo, "sleep\n", 6);
	return;
}

int
count_timer(gpointer data)
{
	PmudData *pd = (PmudData *) data;
	char *tiptext;
	char buffer[100];
	int tmin;

	pd->pmud_status = get_pmud_status(pd);
	if(pd->pmud_status == NULL) {
		return 1;
	}

	if(pd->valid_data == 0) {
		tiptext = malloc(1);
		tiptext[0] = '\0';

		/* The first two flags for left and right seems to the same */
		if(pd->pmud_status->batleft.charging)
			gtk_pixmap_set(GTK_PIXMAP(pd->status_pixmap), pd->charge_in_pixmap, pd->charge_in_pixmap_mask);
		else if(pd->pmud_status->batleft.ac_on)
			gtk_pixmap_set(GTK_PIXMAP(pd->status_pixmap), pd->ac_in_pixmap, pd->ac_in_pixmap_mask);
		else
			gtk_pixmap_set(GTK_PIXMAP(pd->status_pixmap), pd->battery_in_pixmap, pd->battery_in_pixmap_mask);

		// Battery left
		if(pd->pmud_status->batleft.present) {
			gtk_gauge_set_value(GTK_GAUGE(pd->gauge), pd->pmud_status->batleft.current);
			gtk_widget_show(pd->bar1);
			if(pd->pmud_status->batleft.charging)
				tmin = (pd->pmud_status->batleft.maxcharge - pd->pmud_status->batleft.charge) * 66.0 / pd->pmud_status->batleft.current;
			else
				tmin = pd->pmud_status->batleft.charge * 59.2 / -pd->pmud_status->batleft.current;
			if(pd->pmud_status->batleft.charge == pd->pmud_status->batleft.maxcharge)
				tmin = 0;
			sprintf(buffer, "Battery left: %2.2fV, %2.1f%%, %d:%02dh\n", pd->pmud_status->batleft.voltage,
		                     	(float) pd->pmud_status->batleft.charge/pd->pmud_status->batleft.maxcharge*100,
		                     	tmin/60, tmin%60);
			tiptext = realloc(tiptext, strlen(tiptext)+strlen(buffer)+1);
			strcat(tiptext, buffer);

			gtk_progress_bar_update(GTK_PROGRESS_BAR(pd->bar1), (float) pd->pmud_status->batleft.charge/pd->pmud_status->batleft.maxcharge);
		} else
			gtk_widget_hide(pd->bar1);

		// Battery right
		if(pd->pmud_status->batright.present) {
			gtk_gauge_set_value(GTK_GAUGE(pd->gauge), pd->pmud_status->batright.current);
			gtk_widget_show(pd->bar2);
			if(pd->pmud_status->batright.charging)
				tmin = (pd->pmud_status->batright.maxcharge - pd->pmud_status->batright.charge) * 66.0 / pd->pmud_status->batright.current;
			else
				tmin = pd->pmud_status->batright.charge * 59.2 / -pd->pmud_status->batright.current;
			if(pd->pmud_status->batright.charge == pd->pmud_status->batright.maxcharge)
				tmin = 0;
			sprintf(buffer, "Battery right: %2.2fV, %2.1f%%, %d:%02dh\n", pd->pmud_status->batright.voltage,
		                     	(float) pd->pmud_status->batright.charge/pd->pmud_status->batright.maxcharge*100,
		                     	tmin/60, tmin%60);
			tiptext = realloc(tiptext, strlen(tiptext)+strlen(buffer)+1);
			strcat(tiptext, buffer);

			gtk_progress_bar_update(GTK_PROGRESS_BAR(pd->bar2), (float) pd->pmud_status->batright.charge/pd->pmud_status->batright.maxcharge);
		} else
			gtk_widget_hide(pd->bar2);

		if(pd->pmud_status->batleft.present || pd->pmud_status->batright.present) {
			if(pd->show_current)
				gtk_widget_show(pd->gaugeframe);
			else
				gtk_widget_hide(pd->gaugeframe);
		} else
			gtk_widget_hide(pd->gaugeframe);

		gtk_tooltips_set_tip(pd->tooltip, pd->app, tiptext, NULL);
	} else {
		fprintf(stderr, "Try to reopen connection\n");
		pd->pmudinfo = open_pmudinfo(pd);
	}

	return(1);
}

#define IM_BUFSIZE  100
int
open_pmudinfo(PmudData *pd)
{
	int s;
	int ourPort;
	struct sockaddr_in sai;
	struct hostent *he;
	char buf[IM_BUFSIZE];

	if(pd->pmudinfo) {
		close(pd->pmudinfo);
		pd->pmudinfo = 0;
	}

	ourPort = pd->port;
	if (ourPort == 0)
		ourPort = 879;

	s = socket(PF_INET, SOCK_STREAM, 0);
	if (s) {
		he = gethostbyname(pd->servername);
		if (he) {
			sai.sin_family = AF_INET;
			sai.sin_port = htons(ourPort);
			sai.sin_addr = *(struct in_addr*) he->h_addr;

			if (connect(s, (struct sockaddr*) &sai, sizeof(sai)) == 0) {
				pd->pmudinfo = s;
//				fprintf(stderr, "Open connection: %d\n", s);
			} else
				close(s);
		} else
			close(s);
	}

	read(pd->pmudinfo, buf, IM_BUFSIZE);
	return(pd->pmudinfo);
}

static PMUD_STATUS *
get_pmud_status(PmudData *pd)
{
	char buf[IM_BUFSIZE], *ptr1;
	fd_set fs;
	struct timeval tv;
	int n12, n13, n14, n15;
	PMUD_STATUS *status;

	if(pd->pmudinfo) {
		int error;
		FD_ZERO(&fs);
		FD_SET(pd->pmudinfo, &fs);
		tv.tv_sec = 3;
		tv.tv_usec = 0;
	
		if ((select(FD_SETSIZE, &fs, 0, 0, &tv) > 0)) { // && (!feof(pd->pmudinfo))) {
			if(pd->pmud_status)
				free(pd->pmud_status);
			pd->pmud_status = (PMUD_STATUS *) malloc(sizeof(PMUD_STATUS));
			status = pd->pmud_status;
		
			error = 0;
			memset(buf, 0, IM_BUFSIZE);
			error = read(pd->pmudinfo, buf, IM_BUFSIZE);

			error = 0;
			ptr1 = strchr(buf, '{');
			ptr1++; /* Points now on first character after first '{', which are the flags */
			status->batleft.present = ptr1[2] == '1' ? 1 : 0;
			status->batleft.charging = ptr1[1] == '1' ? 1 : 0;
			status->batleft.ac_on = ptr1[0] == '1' ? 1 : 0;
			ptr1 += 3; /* Skip the flags */
			
			/* If left battery is present then ptr1 should point to ' ' and not '}'.
			   We better test for it
			*/
			if(status->batleft.present && (*ptr1 == ' ')) {
				sscanf(ptr1, "%d %d %d %d", &n12, &n13, &n14, &n15);
				status->batleft.charge = n12;
				status->batleft.maxcharge = n13;
				status->batleft.current = (float) n14;
				status->batleft.voltage = n15/1000.0;
			}
			ptr1 = strchr(ptr1, '{'); /* Find the next '{' for rightt battery */
			if(ptr1 == NULL) {  /* This should not happen */
				free(pd->pmud_status);
				return(NULL);
			}
			ptr1++; /* Now points on first flag */
			status->batright.present = ptr1[2] == '1' ? 1 : 0;
			status->batright.charging = ptr1[1] == '1' ? 1 : 0;
			status->batright.ac_on = ptr1[0] == '1' ? 1 : 0;
			ptr1 += 3; /* Skip the flags */

			/* If right battery is present then ptr1 should point to ' ' and not '}'.
			   We better test for it
			*/
			if(status->batright.present && (*ptr1 == ' ')) {
				sscanf(ptr1, "%d %d %d %d", &n12, &n13, &n14, &n15);
				status->batright.charge = n12;
				status->batright.maxcharge = n13;
				status->batright.current = (float) n14;
				status->batright.voltage = n15/1000.0;
			}
		}
		error = write(pd->pmudinfo, "\n", 1);
	} else {
		fprintf(stderr, "Try to reopen connection\n");
		pd->pmudinfo = open_pmudinfo(pd);
		return(NULL);
	}

	return(pd->pmud_status);
}
#undef IM_BUFSIZE

static GtkWidget
*create_monitor(PmudData *pd)
{
	GtkWidget *hbox;
	GtkAdjustment *adj;
	GtkStyle *style;

	style = gtk_widget_get_style(pd->app);

	pd->tooltip=gtk_tooltips_new();
	gtk_tooltips_set_tip (pd->tooltip, pd->app, "lasjf", NULL);

	hbox = gtk_hbox_new(FALSE, 4);
//	label=gtk_label_new(_("Refresh rate (in milli seconds)"));
	adj=(GtkAdjustment *)gtk_adjustment_new(200,0,1000,1000,100,10);
	pd->bar1 = gtk_progress_bar_new_with_adjustment(adj);
	gtk_widget_set_usize(pd->bar1, 4, 16);
  gtk_progress_bar_set_discrete_blocks(GTK_PROGRESS_BAR(pd->bar1), 10);
	gtk_progress_bar_set_orientation(GTK_PROGRESS_BAR(pd->bar1), GTK_PROGRESS_BOTTOM_TO_TOP);
	gtk_box_pack_start(GTK_BOX(hbox),pd->bar1,FALSE,FALSE,0);

	adj=(GtkAdjustment *)gtk_adjustment_new(500,0,1000,1000,100,10);
	pd->bar2 = gtk_progress_bar_new_with_adjustment(adj);
	gtk_widget_set_usize(pd->bar2, 4, 16);
	gtk_progress_bar_set_orientation(GTK_PROGRESS_BAR(pd->bar2), GTK_PROGRESS_BOTTOM_TO_TOP);
	gtk_box_pack_start(GTK_BOX(hbox),pd->bar2,FALSE,FALSE,0);

	pd->battery_in_pixmap = gdk_pixmap_create_from_xpm_d(pd->app->window, &(pd->battery_in_pixmap_mask),
	               &style->bg[GTK_STATE_NORMAL], (gchar **)battery_xpm);
	pd->ac_in_pixmap = gdk_pixmap_create_from_xpm_d(pd->app->window, &(pd->ac_in_pixmap_mask),
	               &style->bg[GTK_STATE_NORMAL], (gchar **)ac_xpm);
	pd->charge_in_pixmap = gdk_pixmap_create_from_xpm_d(pd->app->window, &(pd->charge_in_pixmap_mask),
	               &style->bg[GTK_STATE_NORMAL], (gchar **)charge_xpm);

	pd->status_pixmap = gtk_pixmap_new(pd->battery_in_pixmap, NULL);
	gtk_box_pack_start(GTK_BOX(hbox),pd->status_pixmap,FALSE,FALSE,0);

	pd->gaugeframe = gtk_frame_new(NULL);
	gtk_frame_set_shadow_type(GTK_FRAME(pd->gaugeframe), GTK_SHADOW_IN);
	pd->gauge = gtk_gauge_new();
	gtk_gauge_set_range(GTK_GAUGE(pd->gauge), -1500, 1500);
	gtk_container_add(GTK_CONTAINER(pd->gaugeframe), pd->gauge);
	gtk_widget_set_usize(pd->gaugeframe, 80, 38);
	gtk_box_pack_start(GTK_BOX(hbox),pd->gaugeframe,FALSE,FALSE,0);

	gtk_widget_show(hbox);
	return hbox;
}

static void
bye(GtkWidget *w, gpointer p)
{
	PmudData *pd = (PmudData *) p;
	if(pd->pmudinfo)
		close(pd->pmudinfo);
	applet_widget_gtk_main_quit();
}

void
save_config(gchar *privcfgpath, PmudData *pd) {
	gnome_config_push_prefix (privcfgpath);
	gnome_config_set_int("gpmudmon/general/updatetime", pd->updatetime);
	gnome_config_set_int("gpmudmon/general/width", pd->width);
	gnome_config_set_int("gpmudmon/general/height", pd->height);
	gnome_config_set_int("gpmudmon/server/port", pd->port);
	gnome_config_set_string("gpmudmon/server/name", pd->servername);
	gnome_config_set_int("gpmudmon/server/showcurrent", pd->show_current);
	gnome_config_pop_prefix ();
	gnome_config_sync();
	gnome_config_drop_all();
}

static int
save_session(GtkWidget *w, gchar *privcfgpath, gchar *globcfgpath,  gpointer p) {
	PmudData *pd = (PmudData *) p;

	save_config(privcfgpath, pd);
	return FALSE;
}

static void
applet_pixel_size_changed_cb (GtkWidget *applet, int size, PmudData *pd)
{
//	if(pd->height <= 15)
	gtk_widget_set_usize(pd->globalbox, -1, size);
	gtk_widget_set_usize(pd->bar1, size/4, -1);
	gtk_widget_set_usize(pd->bar2, size/4, -1);
	gtk_widget_set_usize(pd->gaugeframe, size*1.7, size);
}

void
load_config(gchar *privcfgpath, PmudData *pd) {

fprintf(stderr, "Loading config\n");
	gnome_config_push_prefix (privcfgpath);
	pd->updatetime = gnome_config_get_int("gpmudmon/general/updatetime=2000");
	pd->width = gnome_config_get_int("gpmudmon/general/width=310");
	pd->height = gnome_config_get_int("gpmudmon/general/height=70");
	pd->port = gnome_config_get_int("gpmudmon/server/port=879");
	pd->servername = gnome_config_get_string("gpmudmon/server/name=localhost");
	pd->show_current = gnome_config_get_int("gpmudmon/server/showcurrent=1");
	gnome_config_pop_prefix ();
}

static PmudData *
create_pmudmon_widget(GtkWidget *app) {
	PmudData *pd;

	pd = g_new(PmudData, 1);
	pd->app = app;

	load_config(APPLET_WIDGET(app)->privcfgpath, pd);
	pd->globalbox = create_monitor(pd);

	pd->valid_data = 0;
	pd->pmudinfo = 0;
	pd->pmud_status = NULL;
	pd->pmudinfo = open_pmudinfo(pd);
	applet_widget_add(APPLET_WIDGET(pd->app), pd->globalbox);
	gtk_widget_show_all(pd->app);

	pd->tooltip=gtk_tooltips_new();
	gtk_tooltips_set_tip(pd->tooltip, app, "Pmud Monitor", NULL);

	gtk_signal_connect(GTK_OBJECT(app), "delete_event",
			   GTK_SIGNAL_FUNC(bye), NULL);
	gtk_signal_connect(GTK_OBJECT(app), "save_session",
			   GTK_SIGNAL_FUNC(save_session), pd);
	gtk_signal_connect (GTK_OBJECT (app), "change_pixel_size",
	       GTK_SIGNAL_FUNC (applet_pixel_size_changed_cb), pd);


	applet_widget_register_stock_callback(APPLET_WIDGET(app),
					      "sleep",                    
					      GNOME_STOCK_MENU_PREF,
					      _("Goto Sleep"),         
					      GTK_SIGNAL_FUNC(sleep_cb),
					      (gpointer) pd);

	applet_widget_register_stock_callback(APPLET_WIDGET(app),
					      "preference",                    
					      GNOME_STOCK_MENU_PREF,
					      _("Preferences"),         
					      GTK_SIGNAL_FUNC(prefs_cb),
					      (gpointer) pd);

	applet_widget_register_stock_callback(APPLET_WIDGET(app),
					      "about",                    
					      GNOME_STOCK_MENU_ABOUT,
					      _("Info..."),         
					      GTK_SIGNAL_FUNC(about_cb),
					      NULL);

	count_timer(pd);
	pd->timeout = gtk_timeout_add(pd->updatetime, (GtkFunction)count_timer, pd);
	return(pd);
}

static GtkWidget * applet_start_new_applet(const gchar *goad_id, const gchar **params, gint nparams)
{
	GtkWidget *app;
	PmudData *pd;

fprintf(stderr, "applet_start_new_applet: %s\n", goad_id);
	if(strcmp(goad_id, "gpmudmon_applet")) return NULL;

	app = applet_widget_new(goad_id);
	if (!app)
		g_error("Can't create applet!\n");

	pd = create_pmudmon_widget(app);

	return app;
}


int
main(int argc, char *argv[])
{
	char *goad_id;
	GtkWidget *app;
	PmudData *pd;

	/* initialize the i18n stuff */
	textdomain (PACKAGE);
	bindtextdomain (PACKAGE, GNOMELOCALEDIR);

	applet_widget_init("gpmudmon_applet", VERSION, argc, argv, NULL, 0, NULL);
	applet_factory_new("gpmudmon_applet_factory", NULL, applet_start_new_applet);
	goad_id = (char *)goad_server_activation_id();

	if(goad_id && !strcmp(goad_id, "gpmudmon_applet")) {
		app = applet_widget_new("gpmudmon_applet");
		if(!app)
			g_error("Cannot create gpmudmon applet!");

		pd = create_pmudmon_widget(app);
	}

	applet_widget_gtk_main();
	return(0);

}
