/* -*- Mode: C; indent-tabs-mode: nil; c-basic-offset: 4 c-style: "K&R" -*- */

/*---------------------------------------------------------------------------
Descripion:
  s-avg - reports spatial global mean velocity and rms error of a PIV data 
         stream. Eventually subtracts mean value from piv the data.

   Copyright (C) 2002, 2003, 2004 Gerber van der Graaf

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

------------------------------------------------------------------------*/

#include <stdio.h> 
#include <stdlib.h> 
#include <math.h>
#include <glib.h>
#include <gpiv.h>


/* #define PARFILE "savg.par"     */ /* Parameter file name */
#define PARFILE "gpivrc"     /* Parameter file name */
#define USAGE "\
Usage: scale [-f filename][-h | --help] [-p | --print] [-s|-no_s] \n\
             [-v | --version] [-z dx dy] < stdin > stdout \n\
\n\
keys: \n\
-f filename:           files (without .piv extension) instaed of stdin and \n\
                       stdout \n\
-h | --help:           this on-line help \n\
-p | --print:          print parameters to stdout \n\
-s:                    subtracts nothing (0), mean of dx and dy (2) from \n\
                       displacements or velocity estimators \n\
-no_s:                 suppresses subtracting mean from input data \n\
-v | --version:        prints version \n\
-z dx dy:              zero offset of velocities/displacements \n\
"


#ifdef DEBUG
#define USAGE_DEBUG "\
Developers version also contains:  \n\
               [-p_main] \n\
keys: \n\
-p_'function' N; prints data to be generated in the function; the \n\
                   higher N, the more detailed the output. \n\
                   For N = 10, err_vec will exit at the end of the function"
#endif


#define HELP  "\
s-avg reports spatial global mean velocity and rms error of a PIV data stream"

#define RCSID "$Id: s-avg.c,v 2.8 2006/01/31 14:18:04 gerber Exp $"

/*
 * Global variables 
*/
gboolean fname_logic = FALSE;
gboolean print_par = FALSE;
GpivPivData in_data, out_data;

#ifdef DEBUG
/*
 * Variables for development version
*/
int print_main=0;
#endif 



void 
command_args(int argc, char *argv[], 
             char fname[GPIV_MAX_CHARS],
             GpivPostPar * piv_post_par
             )
/* ----------------------------------------------------------------------------
 * Command line argument handling
 */
{
    char c;
    int argc_next;


    while (--argc > 0 && (*++argv)[0] == '-') {
        argc_next=0;

/*
 * argc_next is set to 1 if the next cmd line argument has to be
 * searched for; in case that the command line argument concerns more
 * than one char or cmd line argument needs a parameter
*/
        while (argc_next==0 && (c = *++argv[0])) {
            switch (c) {

            case 'v':
/*
 * Use Revision Control System (RCS) for version
 */
                printf("%s\n", RCSID); 
                exit(0);

            case 'h':
          printf("%s\n", RCSID); 
          printf("%s\n",HELP);
          printf("%s\n",USAGE);
#ifdef DEBUG
          printf ("\n%s\n",USAGE_DEBUG);
#endif
          exit(0);
/*
 * Spatial scaling
 */
            case 's':
                piv_post_par->subtract = atoi(*++argv);
                piv_post_par->subtract_logic = TRUE;
                argc_next = 1;
                break;
                
            case 'f':
                strcpy(fname,*++argv); 
                fname_logic = TRUE;
                argc_next=1;
                --argc;
                break;

            case 'p':
#ifdef DEBUG
                if ((strcmp(*argv,"p_main" ) !=0)) {
#endif
                    print_par = TRUE;
#ifdef DEBUG
                } else if (strcmp("p_main",*argv) == 0) {  
                    print_main=atoi(*++argv);
                    --argc; 
	    argc_next=1;
                }	  
#endif /* DEBUG */
                break;
                
            case 'z':
                piv_post_par->z_off_dx = atof(*++argv);
                piv_post_par->z_off_dy = atof(*++argv);
                fprintf(stderr, "\n000:: z_off_dx=%f", piv_post_par->z_off_dx);
                fprintf(stderr, "  z_off_dy=%f\n", piv_post_par->z_off_dy);
                piv_post_par->z_off_dx_logic = TRUE;
                piv_post_par->z_off_dy_logic = TRUE;
                argc_next = 1;
                --argc;
                --argc;
                argc_next = 1;
                break;

/*
 * negotion of settings
 */            case 'n': 
		if (strcmp("no_s", *argv) == 0) {    /* --- do not subtract mean -*/
                    piv_post_par->subtract = 0;
                    piv_post_par->subtract_logic = TRUE;
                }
                break;


/*
 * long option keys
 */
	    case '-':
		if (strcmp("-help", *argv) == 0) {
                    printf("\n%s", RCSID);
                    printf("\n%s", HELP);
                    printf("\n%s", USAGE);
                    exit(0);
                } else if (strcmp("-print", *argv) == 0) {
		    print_par = TRUE;
                } else if (strcmp("-version", *argv) == 0) {
                    printf("%s\n", RCSID);
                    exit(0);
                } else {
		    gpiv_error("%s: unknown option: %s", RCSID, *argv);
		}
		argc_next = 1;
		break;

            default:
                fprintf (stderr,USAGE);
#ifdef DEBUG
                printf ("\n%s",USAGE_DEBUG);
#endif
                exit(1);
                break;
            }
        }
    }
    
    if(argc != 0) { 
        gpiv_error("%s: %s", RCSID, USAGE);
#ifdef DEBUG
        printf ("\n%s", USAGE_DEBUG);
#endif
        exit(1); 
    }
}



void 
make_fname(char *fname, 
           char *fname_parameter, 
           char *fname_in, 
           char *fname_out
           )
/* ----------------------------------------------------------------------------
 * function to generate filenames
 */
{
    if (fname_logic == TRUE){
        gpiv_error("%s: Filename has to be set", RCSID);
    }
    
    gpiv_io_make_fname(fname, GPIV_EXT_PAR, fname_parameter);
    if (print_par)     printf("# Data parameter file: %s\n",fname_parameter);
    
    gpiv_io_make_fname(fname, GPIV_EXT_PIV, fname_in);
    if (print_par)     printf("# Input data file: %s\n",fname_in);
    
    gpiv_io_make_fname(fname, GPIV_EXT_SA, fname_out);
    if (print_par)     printf("# Output file: %s\n",fname_out);
    
}



int 
main(int argc, 
     char *argv[]
     )
/*-----------------------------------------------------------------------------
*/
{
    char *err_msg = NULL;
    FILE  *fp_par_dat, *fp;
    char fname[GPIV_MAX_CHARS], 
        fname_out[GPIV_MAX_CHARS], 
        fname_parameter[GPIV_MAX_CHARS], 
        fname_in[GPIV_MAX_CHARS];
    char  d_line[GPIV_MAX_LINES][GPIV_MAX_CHARS], 
        c_line[GPIV_MAX_LINES_C][GPIV_MAX_CHARS];
    float mean_dx=0, sdev_dx=0, mean_dy=0, sdev_dy=0;
    /*
 *Take some strange number as initial values for minima and maximuma
 */
    float min_dx=73.0e9, min_dy=73.0e9, max_dx=-73.0e9, max_dy=-73.0e9;
    int nd_lines=0, nc_lines=0, scale=0, ndata=0;

    GpivPostPar piv_post_par, piv_post_par_default;


    gpiv_post_parameters_logic(&piv_post_par, FALSE);
    gpiv_post_default_parameters(&piv_post_par_default, TRUE);
    command_args(argc, argv, fname, &piv_post_par);
    if (print_par) {
        printf("# %s\n# Command line options:\n", RCSID);
        gpiv_post_print_parameters(piv_post_par);
    }
    

    if(fname_logic == TRUE) {
        make_fname(fname, fname_parameter, fname_in, fname_out);
 

/*
 * Prints command line parameters to par-file 
 */
        if ((fp_par_dat = fopen (fname_parameter, "a")) == NULL) {
            gpiv_error("%s: failure opening %s for input\n", RCSID,
                       fname_parameter);
        }
        fprintf(fp_par_dat, "# %s\n# Command line options:\n", RCSID);
        gpiv_post_fprint_parameters(fp_par_dat, piv_post_par);

/*
 * Reading parametes from PARFILE (and writing to data par-file)
 */
        gpiv_scan_parameter(GPIV_POST_PAR_KEY, PARFILE, &piv_post_par, print_par);
        if ((err_msg =
             gpiv_scan_resourcefiles(GPIV_POST_PAR_KEY, &piv_post_par, print_par))
            != NULL)  gpiv_error ("%s: %s", RCSID, err_msg);
        gpiv_post_fprint_parameters(fp_par_dat, piv_post_par);
	fclose(fp_par_dat);
        
        
    } else {
        gpiv_scan_parameter(GPIV_POST_PAR_KEY, PARFILE, &piv_post_par, print_par);
        if ((err_msg =
             gpiv_scan_resourcefiles(GPIV_POST_PAR_KEY, &piv_post_par, print_par))
            != NULL)  gpiv_error ("%s: %s", RCSID, err_msg);
    }
    
    gpiv_post_check_parameters_read(&piv_post_par, piv_post_par_default);
    

/*
 * Check parameters on correct values and adjust belonging variables
 */


/*
 * As in_data.nx or in_data.ny are not known, the input data file will first 
 * be read
*/

    if(fname_logic == TRUE) {
        if ((scale = gpiv_fcount_pivdata(fname_in, &in_data)) == -1) {
            gpiv_error("%s: Failure calling gpiv_count_pivdata", RCSID);
        } else if (print_par) {
            printf("\nmain: scale=%d in_data.nx=%d in_data.ny=%d",
                   scale, in_data.nx, in_data.ny);
        }
    } else {
        if ((scale = gpiv_count_pivdata(&in_data, d_line, &nd_lines)) == -1) {
            gpiv_error("%s: Failure calling gpiv_count_pivdata", RCSID);
        }
    }


/*
 * Now the parameters are known, data  memory can be allocated
 */
    gpiv_alloc_pivdata(&in_data);
    if (piv_post_par.subtract == 1) {
        out_data.nx = in_data.nx;
        out_data.ny = in_data.ny;
        gpiv_alloc_pivdata(&out_data);
    }


/* 
 * Reading input file of piv data 
 */
#ifdef DEBUG
    if (print_main >= 1) fprintf(stderr,"\ncalling (f)gpiv_read_pivdata\n");
#endif

    if(fname_logic == TRUE) {
        if ((err_msg = 
             gpiv_fread_pivdata(fname_in, &in_data, c_line, 
                                             &nc_lines))
            != NULL) gpiv_error ("%s: %s", RCSID, err_msg);
    } else {
        gpiv_read_pivdata(&in_data, d_line, nd_lines, c_line, &nc_lines);
    }

/*
 * Here the function call of the program body
 */
    if (print_par == TRUE) printf("\n");

#ifdef DEBUG
    if (print_main >= 1) fprintf(stderr,"\ncalling gpiv_post_savg\n");
#endif

    gpiv_post_savg(in_data, &out_data, piv_post_par, &ndata);

/*
 * And writing to output 
 */
    if ( piv_post_par.subtract == 1) {
        if (fname_logic == TRUE) {
            if ((err_msg = 
            gpiv_fwrite_pivdata (fname_out, &out_data, c_line,
                                 nc_lines, scale, RCSID))
                != NULL) gpiv_error ("%s: %s", RCSID, err_msg);
        } else {
            gpiv_write_pivdata(&out_data, c_line, nc_lines, scale, RCSID);
        }
    } else {
        if (fname_logic == TRUE) {
            if((fp=fopen(fname_out,"w"))==NULL) { 
                fprintf (stderr,"\n%s: Failure opening %s for output\n",
                         RCSID,fname_out); 
                return 1;
            }
            fprintf(fp, "ndata = %d", ndata);
            fprintf(fp, " mean_dx = %f sdev_dx = %f min_dx = %f max_dx = %f", 
                    mean_dx, sdev_dx, min_dx,  max_dx);
            fprintf(fp, " mean_dy = %f sdev_dy = %f min_dy = %f max_dy = %f\n", 
                    mean_dy, sdev_dy, min_dy,  max_dy);
        } else {
            printf("ndata = %d", ndata);
            printf(" mean_dx = %f sdev_dx = %f min_dx = %f max_dx = %f", 
                   mean_dx, sdev_dx, min_dx,  max_dx);
            printf(" mean_dy = %f sdev_dy = %f min_dy = %f max_dy = %f\n", 
                   mean_dy, sdev_dy, min_dy,  max_dy);
        }
    }
    
/*
 * Freeing allocated memory of matrices 
 */
    gpiv_free_pivdata(&in_data);
    if (piv_post_par.subtract == 1) {
        gpiv_free_pivdata(&out_data);
    }
    
    if (print_par == TRUE) printf("\n");
    exit(0);
}




