/*
 *  Copyright (C) 2005 Kouji TAKAO <kouji@netlab.jp>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <glib/gi18n.h>
#include <gtk/gtk.h>

#include "gpass/configuration.h"
#include "gpass/entry-factory.h"
#include "gpass/error.h"
#include "application.h"
#include "preferences.h"

static GPassViewClass *parent_class = NULL;

enum {
    COLUMN_TYPE,
    COLUMN_LAUNCHER,
    NUM_COLUMNS
};

/***********************************************************
 *
 * Signal handlers
 *
 ***********************************************************/
static void
launcher_on_edited(GtkCellRendererText *renderer, gchar *path, gchar *new_text,
                   gpointer data)
{
    GPassPreferences *self = GPASS_PREFERENCES(data);
    GtkTreeView *launchers;
    GtkListStore *list_store;
    GtkTreeIter iter;
    gboolean result;

    launchers = GTK_TREE_VIEW
        (glade_xml_get_widget(GPASS_VIEW(self)->xml, "launchers"));
    list_store = GTK_LIST_STORE(gtk_tree_view_get_model(launchers));
    result = gtk_tree_model_get_iter_from_string(GTK_TREE_MODEL(list_store),
                                                 &iter, path);
    if (!result) {
        return;
    }
    gtk_list_store_set(list_store, &iter, COLUMN_LAUNCHER, new_text, -1);
}

static void
save_launchers(GPassPreferences *self)
{
    GPassApplication *app;
    GtkTreeView *launchers;
    GtkListStore *list_store;
    GtkTreeIter iter;
    gchar *type;
    gchar *launcher;
    GPassEntryFactoryCursor *cursor;
    GError *error;
    
    launchers = GTK_TREE_VIEW
        (glade_xml_get_widget(GPASS_VIEW(self)->xml, "launchers"));
    list_store = GTK_LIST_STORE(gtk_tree_view_get_model(launchers));
    if (!gtk_tree_model_get_iter_first(GTK_TREE_MODEL(list_store), &iter)) {
        return;
    }
    app = GPASS_APPLICATION(GPASS_VIEW(self)->model);
    cursor = gpass_entry_factory_create_cursor(app->entry_factory);
    while (1) {
        gtk_tree_model_get(GTK_TREE_MODEL(list_store), &iter,
                           COLUMN_TYPE, &type, COLUMN_LAUNCHER, &launcher, -1);
        error = gpass_entry_factory_cursor_seek(cursor, type);
        if (error != NULL) {
            gpass_error_show_and_exit(error);
        }
        g_object_set(cursor, "launcher", launcher, NULL);
        g_free(type);
        g_free(launcher);
        if (!gtk_tree_model_iter_next(GTK_TREE_MODEL(list_store), &iter)) {
            break;
        }
    }
    g_object_unref(cursor);
}

static void
save_environment(GPassPreferences *self)
{
    GPassApplication *app;
    gint max_depth;
    GtkWidget *widget;
    GPassConfiguration *config = gpass_configuration_instance();
    gboolean visible_secrets;
    gboolean lock;
    gint lock_timeout;
    
    app = GPASS_APPLICATION(GPASS_VIEW(self)->model);
    widget = glade_xml_get_widget(GPASS_VIEW(self)->xml, "undo-levels");
    max_depth = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(widget));
    g_object_set(app->command_stack, "max_depth", max_depth, NULL);

    widget = glade_xml_get_widget(GPASS_VIEW(self)->xml, "visible-secrets");
    g_object_get(widget, "active", &visible_secrets, NULL);
    g_object_set(config, "visible-secrets", visible_secrets, NULL);

    widget = glade_xml_get_widget(GPASS_VIEW(self)->xml, "lock");
    g_object_get(widget, "active", &lock, NULL);
    g_object_set(config, "lock", lock, NULL);

    widget = glade_xml_get_widget(GPASS_VIEW(self)->xml, "lock-timeout");
    lock_timeout = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(widget));
    g_object_set(config, "lock_timeout", lock_timeout, NULL);
}

static void
save_preferences(GPassPreferences *self)
{
    save_launchers(self);
    save_environment(self);
}
     
void
gpass_preferences_on_response(GtkWidget *widget, gint response_id,
                              gpointer user_data)
{
    GPassPreferences *self;
    
    gpass_view_self_from_widget(widget, (gpointer **) &self);
    switch (response_id) {
    case GTK_RESPONSE_OK:
        save_preferences(self);
        GPASS_VIEW(self)->result = GPASS_VIEW_RESULT_SUCCEED;
        break;
    default:
        GPASS_VIEW(self)->result = GPASS_VIEW_RESULT_FAILED;
    }
    gpass_view_shutdown_main_loop(GPASS_VIEW(self));
}

/***********************************************************
 *
 * Methods
 *
 ***********************************************************/
static void
gpass_preferences_instance_init(GTypeInstance *instance,
                                       gpointer g_class)
{
    /* GPassPreferences *self = GPASS_PREFERENCES(instance); */
}

static GError *
gpass_preferences_instance_loaded_glade_xml(GPassView *view)
{
    GPassPreferences *self = GPASS_PREFERENCES(view);
    GtkTreeView *launchers;
    GtkTreeViewColumn *column;
    GtkCellRenderer *renderer;
    GtkListStore *list_store;

    launchers = GTK_TREE_VIEW(glade_xml_get_widget(view->xml, "launchers"));
    renderer = gtk_cell_renderer_text_new();
    column = gtk_tree_view_column_new_with_attributes
        (_("Type"), renderer, "text", COLUMN_TYPE, NULL);
    gtk_tree_view_column_set_min_width(column, 100);
    gtk_tree_view_append_column(launchers, column);
    renderer = gtk_cell_renderer_text_new();
    g_object_set(renderer, "editable", TRUE, NULL);
    g_signal_connect(renderer, "edited", G_CALLBACK(launcher_on_edited), self);
    column = gtk_tree_view_column_new_with_attributes
        (_("Launcher"), renderer, "text", COLUMN_LAUNCHER, NULL);
    gtk_tree_view_append_column(launchers, column);
    list_store = gtk_list_store_new(NUM_COLUMNS,
                                    G_TYPE_STRING,  /* type */
                                    G_TYPE_STRING); /* launcher */
    gtk_tree_view_set_model(launchers, GTK_TREE_MODEL(list_store));
    g_object_unref(list_store);
    return parent_class->loaded_glade_xml(view);
}

static void
update_launchers(GPassPreferences *self)
{
    GPassApplication *app;
    GtkTreeView *launchers;
    GtkListStore *list_store;
    GPassEntryFactoryCursor *cursor;
    gchar *type;
    gchar *launcher;
    GtkTreeIter iter;

    launchers = GTK_TREE_VIEW
        (glade_xml_get_widget(GPASS_VIEW(self)->xml, "launchers"));
    list_store = GTK_LIST_STORE(gtk_tree_view_get_model(launchers));
    gtk_list_store_clear(list_store);
    app = GPASS_APPLICATION(GPASS_VIEW(self)->model);
    cursor = gpass_entry_factory_create_cursor(app->entry_factory);
    while (!gpass_entry_factory_cursor_is_done(cursor)) {
        g_object_get(cursor, "type", &type, "launcher", &launcher, NULL);
        if (launcher == NULL) {
            launcher = "";
        }
        gtk_list_store_append(list_store, &iter);
        gtk_list_store_set(list_store, &iter,
                           COLUMN_TYPE, type,
                           COLUMN_LAUNCHER, launcher,
                           -1);
        gpass_entry_factory_cursor_next(cursor);
    }
    g_object_unref(cursor);
}

static void
update_environment(GPassPreferences *self)
{
    GPassApplication *app;
    gint max_depth;
    GtkWidget *widget;
    GPassConfiguration *config = gpass_configuration_instance();
    gboolean visible_secrets;
    gboolean lock;
    gint lock_timeout;
    
    app = GPASS_APPLICATION(GPASS_VIEW(self)->model);
    widget = glade_xml_get_widget(GPASS_VIEW(self)->xml, "undo-levels");
    g_object_get(app->command_stack, "max_depth", &max_depth, NULL);
    gtk_spin_button_set_value(GTK_SPIN_BUTTON(widget), max_depth);

    widget = glade_xml_get_widget(GPASS_VIEW(self)->xml, "visible-secrets");
    g_object_get(config, "visible-secrets", &visible_secrets, NULL);
    g_object_set(widget, "active", visible_secrets, NULL);

    widget = glade_xml_get_widget(GPASS_VIEW(self)->xml, "lock");
    g_object_get(config, "lock", &lock, NULL);
    g_object_set(widget, "active", lock, NULL);

    widget = glade_xml_get_widget(GPASS_VIEW(self)->xml, "lock-timeout");
    g_object_get(config, "lock_timeout", &lock_timeout, NULL);
    g_object_set(widget, "value", (gdouble) lock_timeout, NULL);
}

static GError *
gpass_preferences_instance_run(GPassView *view, GPassViewResult *result)
{
    GPassPreferences *self = GPASS_PREFERENCES(view);

    update_launchers(self);
    update_environment(self);
    return parent_class->run(view, result);
}

static void
gpass_preferences_instance_finalize(GObject *object)
{
    /* GPassPreferences *self = GPASS_PREFERENCES(object); */
    
    G_OBJECT_CLASS(parent_class)->finalize(object);
}

static void
gpass_preferences_class_init(gpointer g_class, gpointer g_class_data)
{
    GObjectClass *gobject_class = G_OBJECT_CLASS(g_class);
    GPassViewClass *view_class = GPASS_VIEW_CLASS(g_class);
    
    parent_class = g_type_class_peek_parent(g_class);
    gobject_class->finalize = gpass_preferences_instance_finalize;
    view_class->loaded_glade_xml = gpass_preferences_instance_loaded_glade_xml;
    view_class->run = gpass_preferences_instance_run;
}

GType
gpass_preferences_get_type(void)
{
    static GType type = 0;
    
    if (type == 0) {
        static const GTypeInfo info = {
            sizeof(GPassPreferencesClass),
            NULL,
            NULL,
            gpass_preferences_class_init,
            NULL,
            NULL,
            sizeof(GPassPreferences),
            0,
            gpass_preferences_instance_init
        };
        
        type = g_type_register_static(GPASS_TYPE_VIEW,
                                      "GPassPreferencesType", &info, 0);
    }
    return type;
}
