# plugs/quote.py
#
#

__copyright__ = 'this file is in the public domain'

from gozerbot.persist import Persist
from gozerbot.nextid import nextid
from gozerbot.commands import cmnds
from gozerbot.examples import examples
from gozerbot.datadir import datadir
from gozerbot.generic import lockdec, rlog
from gozerbot.plughelp import plughelp
from gozerbot.aliases import aliases
from gozerbot.config import config
if config['dbenable']:
    from gozerbot.db import db
from gozerplugs.plugs.karma import karma
import random, re, time, thread, os

plughelp.add('quote', 'manage quotes')

class Quoteitem(object):

    """ object representing a quote """

    def __init__(self, idnr, txt, nick=None, userhost=None, ttime=None):
        self.id = idnr
        self.txt = txt
        self.nick = nick
        self.userhost = userhost
        self.time = ttime

quoteslock = thread.allocate_lock()
locked = lockdec(quoteslock)

class Quotes(Persist):

    """ list of quotes """

    @locked
    def __init__(self, fname):
        Persist.__init__(self, fname)
        if not self.data:
            self.data = []

    def size(self):
        """ return nr of quotes """
        return len(self.data)

    @locked
    def add(self, nick, userhost, quote):
        """ add a quote """
        id = nextid.next('quotes')
        item = Quoteitem(id, quote, nick, userhost, \
time.time())
        self.data.append(item)
        self.save()
        return id

    @locked
    def addnosave(self, nick, userhost, quote, ttime):
        """ add quote but don't call save """
        id = nextid.next('quotes')
        item = Quoteitem(nextid.next('quotes'), quote, nick, userhost, ttime)
        self.data.append(item)
        return id

    @locked
    def delete(self, quotenr):
        """ delete quote with id == nr """
        for i in range(len(self.data)):
            if self.data[i].id == quotenr:
                del self.data[i]
                self.save()
                return 1

    def random(self):
        """ get random quote """
        if not self.data:
            return None
        quotenr = random.randint(0, len(self.data)-1)
        return self.data[quotenr]

    def idquote(self, quotenr):
        """ get quote by id """
        for i in self.data:
            if i.id == quotenr:
                return i

    def whoquote(self, quotenr):
        """ get who quoted the quote """
        for i in self.data:
            if i.id == quotenr:
                return (i.nick, i.time)

    def last(self, nr=1):
        """ get last quote """
        return self.data[len(self.data)-nr:]

    def search(self, what):
        """ search quotes """
        if not self.data:
            return []
        result = []
        andre = re.compile('and', re.I)
        ands = re.split(andre, what)
        got = 0
        for i in self.data:
            for item in ands:
                if i.txt.find(item.strip()) == -1:
                    got = 0
                    break  
                got = 1
            if got:                  
                result.append(i)
        return result

    def searchlast(self, what, nr=1):
        """ search quotes backwards limit to 1"""
        if not self.data:
            return []
        result = []
        andre = re.compile('and', re.I)
        ands = re.split(andre, what)
        got = 0
        for i in self.data[::-1]:
            for item in ands:
                if i.txt.find(item.strip()) == -1:
                    got = 0
                    break  
                got = 1
            if got:                  
                result.append(i)
                got = 0
        return result

class QuotesDb(object):

    """ quotes db interface """

    def size(self):
        """ return nr of quotes """
        result = db.execute(""" SELECT COUNT(*) FROM quotes """)
        return result[0][0]

    def add(self, nick, userhost, quote):
        """ add a quote """
        result = db.execute(""" INSERT INTO quotes(quote, userhost, \
createtime, nick) VALUES (%s, %s, %s, %s) """, (quote, userhost, \
time.time(), nick))
        return result

    def delete(self, quotenr):
        """ delete quote with id == nr """
        result = db.execute(""" DELETE FROM quotes WHERE indx = %s """, \
(quotenr, ))
        return result

    def random(self):
        """ get random quote """
        result = db.execute(""" SELECT indx FROM quotes """)
        indices = []
        if not result:
            return []
        for i in result:
            indices.append(i[0])
        if indices:
            idnr = random.choice(indices)
            return self.idquote(idnr)

    def idquote(self, quotenr):
        """ get quote by id """
        if quotenr == 0:
            return None
        result = db.execute(""" SELECT indx, quote FROM quotes WHERE \
indx = %s """, quotenr)
        if result:
            return Quoteitem(*result[0])

    def whoquote(self, quotenr):
        """ get who quoted the quote """
        result = db.execute(""" SELECT nick, createtime FROM quotes WHERE \
indx = %s """, (quotenr, ))
        if result:
            return result[0]

    def last(self, nr=1):
        """ get last quote """
        result = db.execute(""" SELECT indx, quote FROM quotes ORDER BY \
indx DESC LIMIT %s """, (nr, ))
        res = []
        if result:
            for i in result:
                res.append(Quoteitem(*i))
        return res

    def search(self, what):
        """ search quotes """
        result = db.execute(""" SELECT indx, quote FROM quotes WHERE \
quote LIKE %s """, '%%%s%%' % what)
        res = []
        if result:
            for i in result:
                res.append(Quoteitem(*i))
        return res

    def searchlast(self, what, nr):
        """ search quotes """
        result = db.execute(""" SELECT indx, quote FROM quotes WHERE \
quote LIKE %s ORDER BY indx DESC LIMIT %s """, ('%%%s%%' % what, nr))
        res = []
        if result:
            for i in result:
                res.append(Quoteitem(*i))
        return res

if config['dbenable']:
    quotes = QuotesDb()
else:
    quotes = Quotes(datadir + os.sep + 'quotes')
assert(quotes)

def size():
    """ return number of quotes """
    return quotes.size()

def search(what, queue):
    """ search the quotes """
    rlog(10, 'quote', 'searched for %s' % what)
    result = quotes.search(what)
    for i in result:
        queue.put_nowait("#%s %s" % (i.id, i.txt))

def handle_quoteadd(bot, ievent):
    """ quote-add <txt> .. add a quote """
    if not ievent.rest:
        ievent.missing("<quote>")
        return
    idnr = quotes.add(ievent.nick, ievent.userhost, ievent.rest)
    ievent.reply('quote %s added' % idnr)

cmnds.add('quote-add', handle_quoteadd, ['USER', 'QUOTEADD'], allowqueue=False)
examples.add('quote-add', 'quote-add <txt> .. add quote', 'quote-add mekker')
aliases.data['aq'] = 'quote-add'

def handle_quotewho(bot, ievent):
    """ quote-who <nr> .. show who added a quote """
    try:
        quotenr = int(ievent.args[0])
    except IndexError:
        ievent.missing("<nr>")
        return
    except ValueError:
        ievent.reply("argument must be an integer")
        return
    result = quotes.whoquote(quotenr)
    if not result or not result[0] or not result[1]:
        ievent.reply('no who quote data available')
        return
    ievent.reply('quote #%s was made by %s on %s' % (quotenr, \
result[0], time.ctime(result[1])))

cmnds.add('quote-who', handle_quotewho, ['USER', 'WEB', 'ANON', 'ANONQUOTE'])
examples.add('quote-who', 'quote-who <nr> .. show who quote <nr>', \
'quote-who 1')
aliases.data['wq'] = 'quote-who'

def handle_quotedel(bot, ievent):
    """ quote-del <nr> .. delete quote by id """
    try:
        quotenr = int(ievent.args[0])
    except IndexError:
        ievent.missing('<nr>')
        return
    except ValueError:
        ievent.reply('argument needs to be an integer')
        return
    if quotes.delete(quotenr):
        ievent.reply('quote deleted')
    else:
        ievent.reply("can't delete quote with nr %s" % quotenr)

cmnds.add('quote-del', handle_quotedel, ['QUOTEDEL', 'OPER', 'QUOTE'])
examples.add('quote-del', 'quote-del <nr> .. delete quote', 'quote-del 2')
aliases.data['dq'] = 'quote-del'

def handle_quotelast(bot, ievent):
    """ quote-last .. show last quote """
    search = ""
    try:
        (nr, search) = ievent.args
        nr = int(nr)  
    except ValueError:
        try:
            nr = ievent.args[0]
            nr = int(nr)
        except (IndexError, ValueError):
            nr = 1
            try:
                search = ievent.args[0]
            except IndexError:
                search = ""
    if nr < 1 or nr > 4:
        ievent.reply('nr needs to be between 1 and 4')
        return
    search = re.sub('^d', '', search)
    if search:
        quotelist = quotes.searchlast(search, nr)
    else:
        quotelist = quotes.last(nr)
    if quotelist != None:
        for quote in quotelist:
            qkarma = karma.get('quote %s' % quote.id)
            if qkarma:
                ievent.reply('#%s (%s) %s' % (quote.id, qkarma, quote.txt))
            else:
                ievent.reply('#%s %s' % (quote.id, quote.txt))
    else:
        ievent.reply("can't fetch quote")

cmnds.add('quote-last', handle_quotelast, ['USER', 'WEB', 'ANON', 'ANONQUOTE'])
examples.add('quote-last', 'show last quote', 'quote-last')
aliases.data['lq'] = 'quote-last'

def handle_quote2(bot, ievent):
    """ quote-2 .. show 2 random quotes """
    quote = quotes.random()
    if quote:
        qkarma = karma.get('quote %s' % quote.id)
        if qkarma:
            ievent.reply('#%s (%s) %s' % (quote.id, qkarma, quote.txt))
        else:
            ievent.reply('#%s %s' % (quote.id, quote.txt))
    else:
        ievent.reply('no quotes yet')
        return
    quote = quotes.random()
    if quote:
        qkarma = karma.get('quote %s' % quote.id)
        if qkarma:
            ievent.reply('#%s (%s) %s' % (quote.id, qkarma, quote.txt))
        else:
            ievent.reply('#%s %s' % (quote.id, quote.txt))

cmnds.add('quote-2', handle_quote2, ['USER', 'WEB', 'ANON', 'ANONQUOTE'])
examples.add('quote-2', 'quote-2 .. show 2 random quotes', 'quote-2')
aliases.data['2q'] = 'quote-2'

def handle_quoteid(bot, ievent):
    """ quote-id <nr> .. show quote by id """
    try:
        quotenr = int(ievent.args[0])
    except IndexError:
        ievent.missing('<nr>')
        return
    except ValueError:
        ievent.reply('argument must be an integer')
        return
    quote = quotes.idquote(quotenr)
    if quote:
        qkarma = karma.get('quote %s' % quote.id)
        if qkarma:
            ievent.reply('#%s (%s) %s' % (quote.id, qkarma, quote.txt))
        else:
            ievent.reply('#%s %s' % (quote.id, quote.txt))
    else:
        ievent.reply("can't fetch quote with id %s" % quotenr)

cmnds.add('quote-id', handle_quoteid, ['USER', 'WEB', 'ANON', 'ANONQUOTE'])
examples.add('quote-id', 'quote-id <nr> .. get quote <nr>', 'quote-id 2')
aliases.data['iq'] = 'quote-id'

def handle_quote(bot, ievent):
    """ quote .. show random quote """
    quote = quotes.random()
    if quote:
        qkarma = karma.get('quote %s' % quote.id)
        if qkarma:
            ievent.reply('#%s (%s) %s' % (quote.id, qkarma, quote.txt))
        else:
            ievent.reply('#%s %s' % (quote.id, quote.txt))
    else:
        ievent.reply('no quotes yet')

cmnds.add('quote', handle_quote, ['USER', 'WEB', 'ANON', 'ANONQUOTE'])
examples.add('quote', 'show random quote', 'quote')
aliases.data['q'] = 'quote'

def handle_quotesearch(bot, ievent):
    """ quote-search <txt> .. search quotes """
    if not ievent.rest:
        ievent.missing('<item>')
        return
    else:
        what = ievent.rest
        nrtimes = 3
    result = quotes.search(what)
    if result:
        if ievent.queues:
            res = []
            for quote in result:
                res.append('#%s %s' % (quote.id, quote.txt))
            ievent.reply(res)
            return            
        if nrtimes > len(result):
            nrtimes = len(result)
        randquotes = random.sample(result, nrtimes)
        for quote in randquotes:
            qkarma = karma.get('quote %s' % quote.id)
            if qkarma:
                ievent.reply('#%s (%s) %s' % (quote.id, qkarma, quote.txt))
            else:
                ievent.reply("#%s %s" % (quote.id, quote.txt))
    else:
        ievent.reply('no quotes found with %s' % what)

cmnds.add('quote-search', handle_quotesearch, ['USER', 'WEB', 'ANON', \
'ANONQUOTE'])
examples.add('quote-search', 'quote-search <txt> .. search quotes for <txt>'\
, 'quote-search bla')
aliases.data['sq'] = 'quote-search'

def handle_quotescount(bot, ievent):
    """ quote-count .. show number of quotes """
    ievent.reply('quotes count is %s' % quotes.size())

cmnds.add('quote-count', handle_quotescount, ['USER', 'WEB', 'ANON', \
'ANONQUOTE'])
examples.add('quote-count', 'count nr of quotes', 'quote-count')
aliases.data['cq'] = 'quote-count'

def handle_quotegood(bot, ievent):
    """ show top ten positive karma """
    result = karma.quotegood(limit=10)
    if result:
        resultstr = ""
        for i in result:
            if i[1] > 0:
                resultstr += "%s: %s " % (i[0], i[1])
        ievent.reply('quote goodness: %s' % resultstr)
    else:
        ievent.reply('quote karma void')

cmnds.add('quote-good', handle_quotegood, ['USER', 'WEB', 'ANON', 'ANONQUOTE'])
examples.add('quote-good', 'show top 10 quote karma', 'quote-good')

def handle_quotebad(bot, ievent):
    """ show top ten negative karma """
    result = karma.quotebad(limit=10)
    if result:
        resultstr = ""
        for i in result:
            if i[1] < 0:
                resultstr += "%s: %s " % (i[0], i[1])
        ievent.reply('quote badness: %s' % resultstr)
    else:
        ievent.reply('quote karma void')

cmnds.add('quote-bad', handle_quotebad, ['USER', 'WEB', 'ANON', 'ANONQUOTE'])
examples.add('quote-bad', 'show lowest 10 quote karma', 'quote-bad')
