/* GnomerMind - A MasterMind(R)-based game for Gnome
 * (C) 2001 Germano Rizzo
 *
 * gnomermind.c - game managing functions implementation
 * Author: Germano Rizzo
 *
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "common.h"
#include <string.h>
#include "face.h"
#include "menu.h"
#include "themes.h"
#include "sounds.h"
#include "save.h"
#include "cache.h"

const gchar *app = "GnomerMind";

gint try[MAXR][MAXH], field[MAXH], b[MAXR], w[MAXR];
gint state, colsel, act;
gboolean drop, inf, rep, bla;

gint colorz = 0;
gint holez = 0;
gint roundz = 0;
gchar *geometry = NULL;

/*
 * these are defined and used only for the commandline parsing
 */

static gboolean override = FALSE;
static gchar *dr = NULL;
static gchar *as = NULL;
static gchar *in = NULL;
static gchar *re = NULL;
static gchar *bl = NULL;

/*
 * this is the commandline argument parsing items vector
 */

static struct poptOption options[] = {
  {
   "places",
   'p',
   POPT_ARG_INT,
   &holez,
   0,
   N_("Number of tokens to guess (4-6, default 4)"),
   N_("TOKENS")},
  {
   "colors",
   'c',
   POPT_ARG_INT,
   &colorz,
   0,
   N_("Number of different tokens (4-10, default 6)"),
   N_("RANGE")},
  {
   "rounds",
   'r',
   POPT_ARG_INT,
   &roundz,
   0,
   N_("Number of available rounds (6-12, default 8)"),
   N_("ROUNDS")},
  {
   "infinite",
   'i',
   POPT_ARG_STRING,
   &in,
   0,
   N_("Play on infinite rounds (default no)"),
   "<yes/no>"},
  {
   "repetitions",
   'd',
   POPT_ARG_STRING,
   &re,
   0,
   N_
   ("Allow repetitions of tokens of different types in the row to guess (default yes)"),
   "<yes/no>"},
  {
   "blank-places",
   'b',
   POPT_ARG_STRING,
   &bl,
   0,
   N_
   ("Allow to leave blank places in guessings; they won't be evaluated (default no)"),
   "<yes/no>"},
  {
   "theme",
   '\0',
   POPT_ARG_STRING,
   &theme,
   0,
   N_
   ("Specifies a theme to use, or the path of its directory if not installed (default \"Classic\")"),
   N_("THEME")},
  {
   "override",
   '\0',
   POPT_ARG_NONE,
   &override,
   0,
   N_("Ignore the saved game (if any)"),
   NULL},
  {
   "drop",
   '\0',
   POPT_ARG_STRING,
   &dr,
   0,
   N_("Drop the token after each move (default no)"),
   "<yes/no>"},
  {
   "confirmations",
   '\0',
   POPT_ARG_STRING,
   &as,
   0,
   N_("Turn on the confirmation dialogs (default yes)"),
   "<yes/no>"},
  {
   "geometry",
   '\0',
   POPT_ARG_STRING,
   &geometry,
   0,
   N_("Specify the geometry of the main window"),
   "WxH+X+Y"},
  {
   NULL,
   '\0',
   0,
   NULL,
   0,
   NULL,
   NULL}
};

/**
 * getrnd
 * 
 * Generates a random value
 **/
static inline gint
getrnd ()
{
  static gint ret = 0;
  FILE *fr;

  fr = fopen ("/dev/urandom", "r");

  /*
   * if /dev/urandom does not exist, it falls back on
   * returning the current time
   */

  if (!fr)
    {
      if (!ret)
	ret = time (NULL);
      srandom (ret);
      ret = random ();

      return ret;
    }

  fread (&ret, 1, 1, fr);
  fclose (fr);

  return ret;
}

/**
 * initfield
 *
 * Generates the field to guess, resetting also the parameters
 **/
static void
initfield ()
{
  gint i;
  gboolean used_col[MAXC];

  g_assert (rep || (colorz >= holez));

  if (!rep)
    for (i = 0; i < MAXC; i++)
      used_col[i] = FALSE;

  for (i = 0; i < holez; i++)
    {
      gint val;

      val = getrnd () % colorz + 1;

      g_assert (val > 0 && val <= colorz);

      if (rep || !used_col[val - 1])
	{
	  field[i] = val;
	  used_col[val - 1] = TRUE;
	}
      else
	i--;
    }

  act = 0;
  colsel = 0;
  state = 0;
}

/**
 * initrow
 * @row: the row to initialize
 *
 * Initializes a row
 **/
static void
initrow (gint row)
{
  gint i;

  b[row] = 0;
  w[row] = 0;
  for (i = 0; i < holez; i++)
    try[row][i] = 0;
}

/**
 * inittry
 *
 * Resets the `try' field
 **/
static void
inittry ()
{
  gint i;

  for (i = 0; i < roundz; i++)
    initrow (i);
}

/**
 * rst
 * @reply: the pressed button
 * @data: ignored, only for callback
 *
 * Resets all but the field to guess
 **/
void
rst (gint reply, gpointer data)
{
  if (reply == GNOME_YES)
    {
      if (state > 1)
	solution_hide ();
      act = 0;
      state = 0;
      colsel = 0;
      inittry ();
      greset ();
      preset ();
    }
}

/**
 * newall
 * @reply: the pressed button
 * @data: ignored, only for callback
 *
 * Begins all over again
 **/
void
newall (gint reply, gpointer data)
{
  if (reply == GNOME_YES)
    {
      if (state > 1)
	solution_hide ();
      inittry ();
      initfield ();
      greset ();
      preset ();
    }
}

/**
 * suicide
 * @reply: the pressed button
 * @data: ignored, only for callback
 *
 * Terminates a game abruptly
 **/
void
suicide (gint reply, gpointer data)
{
  if (reply == GNOME_YES)
    {
      state = 3;
      preset ();
      solution (FALSE);
    };
}

/**
 * newngr
 *
 * Resets all but the gfx
 **/
void
newngr ()
{
  inittry ();
  initfield ();
}

/**
 * drawback
 *
 * Shift the field of a place down
 * Called after the last row is completed,
 * to obtain infinite rounds
 **/
static void
drawback ()
{
  gint i, j;

  for (i = 0; i < roundz - 1; i++)
    {
      b[i] = b[i + 1];
      w[i] = w[i + 1];
      for (j = 0; j < holez; j++)
	try[i][j] = try[i + 1][j];
    }

  initrow (roundz - 1);
}

/**
 * count
 * 
 * Counts how many holes in a row are occupied by a token
 **/
static gint
count ()
{
  gint i, c;

  c = 0;
  for (i = 0; i < holez; i++)
    if (try[act][i])
      c++;

  g_assert (c >= 0 && c <= holez);

  return c;
}

/**
 * control
 *
 * Evaluates the player's guessing, and sets the field properly
 **/
static void
control ()
{
  gint i, j, c;
  gboolean fmask[holez], tmask[holez];

  c = count ();

  /*
   * a token, either in the field or in the guess row, must be evaluated
   * one time only. So, we keep two vector to tell wheter it's yet been
   * evaluated or not; fmask for the field, tmask for the guess
   */

  for (i = 0; i < holez; i++)
    {
      fmask[i] = TRUE;
      tmask[i] = TRUE;
    }

  msgpop ();

  /*
   * first, we check for black pins, and set off both the masks if we
   * found any 
   */

  for (i = 0; i < holez; i++)
    if (try[act][i] == field[i])
      {
	b[act]++;
	fmask[i] = FALSE;
	tmask[i] = FALSE;
      }

  /*
   * then, the white ones
   */

  for (i = 0; i < holez; i++)
    for (j = 0; j < holez; j++)
      if ((tmask[j]) && (fmask[i]) && (try[act][j] == field[i]))
	{
	  w[act]++;
	  fmask[i] = FALSE;
	  tmask[j] = FALSE;
	}

  putpins (roundz - act - 1, b[act], w[act]);

  g_assert (b[act] <= holez && b[act] >= 0);
  g_assert (w[act] <= holez && w[act] >= 0);
  g_assert (b[act] + w[act] <= holez && b[act] + w[act] >= 0);

  if (b[act] == holez)
    {

      /*
       * success!
       */

      state = 2;
      preset ();
      colsel = 0;
      solution (TRUE);
    }
  else
    {
      if (act == roundz - 1)
	{

	  /*
	   * we've reached the last row
	   */

	  if (inf)
	    {
	      /*
	       * infinite rounds
	       */

	      msgpush (g_strdup_printf
		       (_("%d correct, %d misplaced, %d wrong"), b[act],
			w[act], c - b[act] - w[act]));
	      drawback ();
	      retok ();
	      state = 0;
	      preset ();
	    }
	  else
	    {

	      /*
	       * failure
	       */

	      state = 3;
	      preset ();
	      solution (FALSE);
	    }
	}
      else
	{

	  /*
	   * normal move
	   */

	  msgpush (g_strdup_printf
		   (_("%d correct, %d misplaced, %d wrong"), b[act], w[act],
		    c - b[act] - w[act]));
	  act++;
	  state = 0;
	  initrow (act);
	  preset ();
	  colsel = 0;
	}
    }
}

/**
 * manage
 * @cmd: the type of the command to evaluate
 * @col: the selected column or color (if appl.)
 * @row: the selected row (if appl.)
 *
 * Called by the frontend to manage a move
 **/
void
manage (gint cmd, gint col, gint row)
{
  switch (cmd)
    {
    case 0:
      {
#ifdef HAVE_ESD
	sound_activate ();
#endif

	/*
	 * row==-1  autoselects the current row
	 */

	if (row == -1)
	  row = roundz - act - 1;

	/*
	 * click on a not-yet-evaluated row hole
	 */

	if ((act == roundz - row - 1) && (state < 2))
	  {

	    /*
	     * click on an occupied hole
	     */

	    if ((try[act][col] > 0) && (!colsel))
	      {
		gint c;

		colsel = try[act][col];
		puttok (col, row, 0);
		try[act][col] = 0;
		pick (colsel, FALSE);
		c = count ();
		if ((!bla && (c < holez)) || (bla && !c))
		  {
		    state = 0;
		    chstate ();
		  }
	      }
	    else
	      {

		/*
		 * click on an empty hole
		 */

		if (colsel)
		  {
		    gint c;

		    try[act][col] = colsel;
		    puttok (col, row, colsel);
		    if (drop)
		      {
			pick (colsel, TRUE);
			colsel = 0;
		      }
		    c = count ();
		    if ((c == holez) || (bla && c))
		      {
			state = 1;
			chstate ();
		      }
		  }
	      }
	  }

	/*
	 * click on a yet-evaluated "back row" hole
	 */

	else if (act > roundz - row - 1)
	  if (try[roundz - row - 1][col])
	    {
	      gint oldcol = colsel;

	      oldcol = colsel;
	      colsel = try[roundz - row - 1][col];
	      pick (colsel, FALSE);
	      if (oldcol && (colsel != oldcol))
		pick (oldcol, TRUE);
	    }
      }
      break;
    case 1:
      {
#ifdef HAVE_ESD
	sound_activate ();
#endif
	/*
	 * click on a palette hole
	 */

	if (state < 2)
	  {
	    if (colsel)
	      {
		gint oldcol = colsel;

		if (colsel == col)
		  colsel = 0;
		else
		  pick (colsel = col, FALSE);

		pick (oldcol, TRUE);
	      }
	    else
	      pick (colsel = col, FALSE);
	  }
      }
      break;
    case 2:
      {

	/*
	 * click on the action icon
	 */

	switch (state)
	  {
	  case 0:
	    break;
	  case 1:
	    {
#ifdef HAVE_ESD
	      sound_activate ();
#endif
	      control ();
	    }
	    break;
	  case 2:
	  case 3:
	    {
	      solution_hide ();
	      newall (GNOME_YES, NULL);
	    }
	    break;
	  default:
	    g_assert_not_reached ();
	  }
      }
      break;
    default:
      g_assert_not_reached ();
    }
}

/**
 * direct_sel
 * @col: the chosen color
 *
 * Manages a `direct' (usually right-click) selection
 **/
void
direct_sel (gint col)
{
  gint i, oldcolsel;

  if (col == colsel)
    return;

  oldcolsel = colsel;

  for (i = 0; i < holez; i++)
    if (!try[act][i])
      {
	colsel = col;
	manage (0, i, -1);
	colsel = oldcolsel;
	return;
      }
}

/**
 * in_use
 * @col: the color to search
 *
 * Checks if a color is already used
 **/
static inline gboolean
in_use (gint col)
{
  gint i;

  for (i = 0; i < holez; i++)
    if (try[act][i] == col)
      return TRUE;

  return FALSE;
}

/**
 * rnd_sel
 * @x: the hole
 *
 * Manages a `random' selection
 **/
void
rnd_sel (gint x)
{
  gint oldcol;

  oldcol = colsel;
  colsel = getrnd () % colorz + 1;

  if (!rep)
    while (in_use (colsel))
      colsel = getrnd () % colorz + 1;

  manage (0, x, -1);

  colsel = oldcol;
}

/**
 * save_yourself
 *
 * Callback for the session management
 * taken from the standard GNOME documentation
 **/
static gint
save_yourself (GnomeClient * client, gint phase, GnomeSaveStyle save_style,
	       gint is_shutdown, GnomeInteractStyle interact_style,
	       gint is_fast, gpointer client_data)
{
  gchar *args[4] = { "rm", "-r", NULL, NULL };

  current_save ();

  args[2] = gnome_config_get_real_path
    (gnome_client_get_config_prefix (client));

  gnome_client_set_discard_command (client, 3, args);

  return TRUE;
}

/**
 * main
 * @argc: standard main() parameter
 * @argv: standard main() parameter
 *
 * The main routine
 **/
int
main (int argc, char *argv[])
{
  gboolean isdefault;
  GnomeClient *client;

#ifdef ENABLE_NLS
  bindtextdomain (PACKAGE, GNOMELOCALEDIR);
  textdomain (PACKAGE);
#endif

  gnome_init_with_popt_table (PACKAGE, VERSION, argc, argv, options, 0, NULL);
  gdk_rgb_init ();

#ifndef G_DISABLE_ASSERT
  g_message ("Running with g_assertions on");
#endif

  /*
   * the session management stuff
   */

  client = gnome_master_client ();
  gtk_signal_connect (GTK_OBJECT (client), "save_yourself",
		      GTK_SIGNAL_FUNC (save_yourself), NULL);
  gtk_signal_connect (GTK_OBJECT (client), "die",
		      GTK_SIGNAL_FUNC (gtk_main_quit), NULL);

  /*
   * various validity checks for the commandline parameters
   */

  g_assert (MAXH <= 6);
  g_assert (MAXC <= 10);

  g_assert (MINH > 0);
  g_assert (MINC > 0);
  g_assert (MINR > 0);

  g_assert (MINH < MAXH);
  g_assert (MINC < MAXC);
  g_assert (MINR < MAXR);

  g_assert (MINH <= DEFH && DEFH <= MAXH);
  g_assert (MINC <= DEFC && DEFC <= MAXC);
  g_assert (MINR <= DEFR && DEFR <= MAXR);

  colorz = normalize_popt_int (colorz, MINC, MAXC);
  holez = normalize_popt_int (holez, MINH, MAXH);
  roundz = normalize_popt_int (roundz, MINR, MAXR);

  if (in)
    if (!validate_popt_string (in))
      in = NULL;

  if (as)
    if (!validate_popt_string (as))
      as = NULL;

  if (dr)
    if (!validate_popt_string (dr))
      dr = NULL;

  if (re)
    if (!validate_popt_string (re))
      re = NULL;

  if (bl)
    if (!validate_popt_string (bl))
      bl = NULL;

  isdefault = parameter_load_with_commandline (in, re, bl, override, dr, as);

  theme_init ();

  g_assert (theme != NULL);

  if (strrchr (theme, G_DIR_SEPARATOR))
    {
      gchar *th, *pa;
      gint len;

      len = strlen (theme);

      while (theme[len - 1] == G_DIR_SEPARATOR)
	theme[--len] = '\0';

      th = strrchr (theme, G_DIR_SEPARATOR);
      th++;

      len = strlen (theme) - strlen (th);
      pa = g_strndup (theme, len);
      theme_path = g_strdup (pa);
      theme = g_strdup (th);
      newtheme = theme;
    }

  if (!validate (theme))
    {
      theme_path = THEMESPATH;
      if (validate ("Classic"))
	theme = "Classic";
      else
	theme = g_strdup (g_list_first (theme_list)->data);
    }

  g_assert (validate (theme));
  g_assert (roundz <= MAXR && roundz >= MINR);
  g_assert (colorz <= MAXC && colorz >= MINC);
  g_assert (holez <= MAXH && holez >= MINH);
  g_assert (state <= 3 && state >= 0);

  /*
   * cache loading
   */

  theme_cache_load ();

#ifdef HAVE_ESD
  sound_cache_load ();
#endif

  /*
   * draws the board
   */

  build_board ();

  /*
   * game starts
   */

  if (isdefault)
    newngr ();
  else
    retok ();

  gtk_main ();

  return (0);
}
