/*  GnomeKiss - A KiSS viewer for the GNOME desktop
    Copyright (C) 2000-2001  Nick Lamb <njl195@zepler.org.uk>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gnome.h>
#include <limits.h>
#include <sys/time.h>
#include <unistd.h>

#include "kiss.h"

guint fk_timer_source= 0;
MicroTimer *timers = NULL; /* Timers ready to fire */

/* internals */

static gboolean check_timers(gpointer data);
static MicroTimer timer_zero;
static struct timeval lasttime;

static void unset_timer(KissActionList alarm);

static gboolean check_timers(gpointer data) {
  KissActionList actions;
  struct timeval thistime;
  long count;

  gettimeofday(&thistime, NULL);

  count= ( thistime.tv_sec - lasttime.tv_sec ) * 1000;
  count+= ( thistime.tv_usec - lasttime.tv_usec ) / 1000;
  count/= prefs.speed_factor;

  lasttime.tv_sec= thistime.tv_sec;
  lasttime.tv_usec= thistime.tv_usec;

  while (timers->what != NULL && timers->when < count) {
    count-= timers->when;
    actions= timers->what;
    timer_zero.next= timers->next;
    timers->next->prev= &timer_zero;
    g_free(timers);
    timers= timer_zero.next;
    events(actions);
  }

  if (timers->what) {
    timers->when-= count;
    return TRUE;
  } else {
    fk_timer_source= 0; /* source gone */
    return FALSE;
  }
}

void set_timer(int which, long when) {
  long count= 0;
  MicroTimer *timer, *new_timer;
  KissActionList what;

  if (which < 0 || which >= ALARMS) {
    fprintf(stderr, _("Attempt to set or cancel alarm %d ignored\n"), which);
    return; /* not valid */
  } else {
    what= config.alarms[which];
  }

  if (what == NULL) {
    return; /* nothing to do, so who cares when it would have happened */
  }

  unset_timer(what); /* Wasteful but easiest to code it this way */

  if (when == 0) {
    return; /* removing the old timer was all we needed to do */
  }

  if (when % prefs.min_timer > 0) {
    when+= (prefs.min_timer - (when % prefs.min_timer));
  }

  if (fk_timer_source == 0) {
    fk_timer_source= gtk_timeout_add(10, (GSourceFunc) check_timers, NULL);
    gettimeofday(&lasttime, NULL);
  }

  new_timer= g_new(MicroTimer, 1);
  new_timer->what= what;

  for (timer= timers; timer != NULL; timer=timer->next) {
    count+= timer->when;

    if (new_timer && count >= when) {
      new_timer->when= when + timer->when - count;
      new_timer->prev= timer->prev;
      new_timer->next= timer;
      timer->when= count - when;
      timer->prev->next= new_timer;
      timer->prev= new_timer;
      new_timer= NULL;
    }

    if (timer->what == NULL) { /* timer_zero */
      timer->when= LONG_MAX / 2;
      break;
    }
  }

  timers= timer_zero.next;
}

static void unset_timer(KissActionList alarm) {
  MicroTimer *timer, *old_timer = NULL;

  if (alarm == NULL) {
    return;
  }
  for (timer= timers; timer != NULL; timer=timer->next) {
    if (timer->what == alarm) {
      old_timer= timer;
      timer->prev->next= timer->next;
      timer->next->when+= timer->when;
      timer->next->prev= timer->prev;
      timer= timer->prev;
      g_free(old_timer); /* clean up */
    }
    if (timer->what == NULL) { /* timer_zero */
      timer->when= LONG_MAX / 2;
      break;
    }
  }
  timers= timer_zero.next;
}

void reset_timers(void) {
  MicroTimer *next;
  
  while (timers && timers->what) {
    next= timers->next;
    g_free(timers);
    timers= next;
  }

  timer_zero.when= LONG_MAX / 2;
  timer_zero.what= NULL;
  timer_zero.next= &timer_zero;
  timer_zero.prev= &timer_zero;
  timers= timer_zero.next;

  if (fk_timer_source > 0) {
    g_source_remove(fk_timer_source);
    fk_timer_source= 0;
  }
}
