/*
    GNOME Commander - A GNOME based file manager 
    Copyright (C) 2001-2003 Marcus Bjurman

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/ 
#include <config.h>
#include <sys/types.h>
#include <regex.h>
#include <unistd.h>
#include <errno.h>
#include "gnome-cmd-includes.h"
#include "gnome-cmd-bookmark-dialog.h"
#include "gnome-cmd-main-win-funcs.h"
#include "gnome-cmd-file-selector-funcs.h"
#include "gnome-cmd-file-funcs.h"
#include "gnome-cmd-clist.h"
#include "gnome-cmd-dir-funcs.h"
#include "gnome-cmd-string-dialog.h"
#include "gnome-cmd-data.h"
#include "gnome-cmd-combo.h"
#include "gnome-cmd-ftp-server-funcs.h"
#include "gnome-cmd-connection-funcs.h"
#include "support.h"
#include "widget-factory.h"


struct _GnomeCmdBookmarkDialogPrivate
{
	GList *groups;
	GnomeCmdBookmark *sel_bookmark;
	GnomeCmdBookmarkGroup *sel_group;

	GtkWidget *combo;
	GtkWidget *dir_list;
	GtkWidget *remove_btn;
	GtkWidget *edit_btn;
	GtkWidget *goto_btn;		
	GtkWidget *move_up_btn;		
	GtkWidget *move_down_btn;		
};


static GnomeDialogClass *parent_class = NULL;

guint bookmark_dialog_default_column_width[BOOKMARK_DIALOG_NUM_COLUMNS] = {
	16,
	100,
	300,
};

static void
show_bookmark_dialog (const gchar *name, const gchar *path,
					  const gchar *title,
					  GnomeCmdStringDialogCallback on_ok,
					  GnomeCmdBookmarkDialog *dialog)
{
	const gchar *labels[] = {
		_("Bookmark name:"),
		_("Bookmark target:"),
	};
	
	GtkWidget *dlg =
		gnome_cmd_string_dialog_new (
			title,
			labels,
			2,
			(GnomeCmdStringDialogCallback)on_ok,
			dialog);
	
	gnome_cmd_string_dialog_set_value (GNOME_CMD_STRING_DIALOG (dlg), 0, name);
	gnome_cmd_string_dialog_set_value (GNOME_CMD_STRING_DIALOG (dlg), 1, path);
	
	gtk_widget_show (dlg);
}


static GnomeCmdConnection *
get_connection_by_bookmark (GnomeCmdBookmark *bookmark, gboolean *using_old)
{
	GList *cons = gnome_cmd_connection_get_all_connections ();
	GnomeCmdFtpServer *server = bookmark->group->server;
	GnomeCmdConnection *con;

	*using_old = TRUE;
	
	while (cons) {
		con = (GnomeCmdConnection*)cons->data;
		if (con->type == CON_TYPE_FTP && con->server == server)
			return con;
		else if (con->type == CON_TYPE_LOCAL && !server)
			return con;
		cons = cons->next;
	}

	if (server) {
		con = gnome_cmd_connection_new_ftp_from_server (server, bookmark->path);

		gnome_cmd_file_selector_update_connections (
			gnome_cmd_main_win_get_active_fs (main_win));
		gnome_cmd_file_selector_update_connections (
			gnome_cmd_main_win_get_inactive_fs (main_win));

		*using_old = FALSE;

		return con;
	}
	
	return NULL;
}


static void
do_add_bookmark (GnomeCmdBookmarkDialog *dialog, GnomeCmdBookmark *bookmark)
{
	gint row;
	gchar *text[4];

	g_return_if_fail (bookmark != NULL);

	text[0] = NULL;
	text[1] = bookmark->name;
	text[2] = bookmark->path;
	text[3] = NULL;

	row = gtk_clist_append (GTK_CLIST (dialog->priv->dir_list), text);
	gtk_clist_set_row_data (GTK_CLIST (dialog->priv->dir_list), row, bookmark);

	gtk_clist_set_pixmap (GTK_CLIST (dialog->priv->dir_list), row, 0,
						  IMAGE_get_pixmap (PIXMAP_BOOKMARK),
						  IMAGE_get_mask (PIXMAP_BOOKMARK));
}


static void
set_selected_group (GnomeCmdBookmarkDialog *dialog, GnomeCmdBookmarkGroup *group)
{
	GList *bookmarks;

	g_return_if_fail (group != NULL);
	
	dialog->priv->sel_group = group;

	gtk_entry_set_text (GTK_ENTRY (GNOME_CMD_COMBO (dialog->priv->combo)->entry), group->alias);

	gtk_clist_clear (GTK_CLIST (dialog->priv->dir_list));
	bookmarks = group->bookmarks;
	while (bookmarks) {
		GnomeCmdBookmark *bookmark = (GnomeCmdBookmark*)bookmarks->data;
		do_add_bookmark (dialog, bookmark);
		bookmarks = bookmarks->next;
	}
}


static void on_dir_goto (GtkButton *button, GnomeCmdBookmarkDialog *dialog)
{
	GnomeCmdBookmark *bookmark;
	GtkCList *dir_list = GTK_CLIST (dialog->priv->dir_list);

	bookmark = (GnomeCmdBookmark*)gtk_clist_get_row_data (dir_list, dir_list->focus_row);

	gnome_cmd_bookmark_goto (bookmark);
	
	gtk_widget_destroy (GTK_WIDGET (dialog));
}


static void on_dir_remove (GtkButton *button, GnomeCmdBookmarkDialog *dialog)
{
	GnomeCmdBookmark *bookmark;
	GtkCList *dir_list = GTK_CLIST (dialog->priv->dir_list);
	
	bookmark = (GnomeCmdBookmark*)gtk_clist_get_row_data (dir_list, dir_list->focus_row);
	gtk_clist_remove (dir_list, dir_list->focus_row);

	bookmark->group->bookmarks = g_list_remove (bookmark->group->bookmarks, bookmark);
	g_free (bookmark->name);
	g_free (bookmark->path);
	g_free (bookmark);
}


static gboolean on_edit_ok (GnomeCmdStringDialog *string_dialog,
							const gchar **values,
							GnomeCmdBookmarkDialog *dialog)
{
	const gchar *name = values[0];
	const gchar *path = values[1];
	GtkCList *dir_list = GTK_CLIST (dialog->priv->dir_list);

	g_return_val_if_fail (dialog->priv->sel_bookmark != NULL, TRUE);
	
	if (!name) {
		gnome_cmd_string_dialog_set_error_desc (
			string_dialog, g_strdup (_("Bookmark name is missing")));
		return FALSE;
	}
	
	if (!path) {
		gnome_cmd_string_dialog_set_error_desc (
			string_dialog, g_strdup (_("Bookmark target is missing")));
		return FALSE;
	}
	
	g_free (dialog->priv->sel_bookmark->name);
	dialog->priv->sel_bookmark->name = g_strdup (name);
	g_free (dialog->priv->sel_bookmark->path);
	dialog->priv->sel_bookmark->path = g_strdup (path);
	
	gtk_clist_set_text (dir_list, dir_list->focus_row, 1, name);
	gtk_clist_set_text (dir_list, dir_list->focus_row, 2, path);

	return TRUE;
}


static void on_dir_edit (GtkButton *button, GnomeCmdBookmarkDialog *dialog)
{
	show_bookmark_dialog (dialog->priv->sel_bookmark->name,
						  dialog->priv->sel_bookmark->path,
						  _("Edit Bookmark"),
						  (GnomeCmdStringDialogCallback)on_edit_ok,
						  dialog);
}


static void update_move_buttons (GnomeCmdBookmarkDialog *dialog, int row)
{
	GtkCList *dir_list = GTK_CLIST (dialog->priv->dir_list);
	
	if (row == 0) {
		gtk_widget_set_sensitive (dialog->priv->move_up_btn, FALSE);
		gtk_widget_set_sensitive (
			dialog->priv->move_down_btn,
			dir_list->rows > 1);
	}
	else if (row == dir_list->rows - 1) {
		gtk_widget_set_sensitive (dialog->priv->move_down_btn, FALSE);
		gtk_widget_set_sensitive (
			dialog->priv->move_up_btn,
			dir_list->rows > 1);
	}
	else {
		gtk_widget_set_sensitive (dialog->priv->move_up_btn, TRUE);
		gtk_widget_set_sensitive (dialog->priv->move_down_btn, TRUE);
	}
}


static void on_dir_move_up (GtkButton *button, GnomeCmdBookmarkDialog *dialog)
{
	GtkCList *dir_list = GTK_CLIST (dialog->priv->dir_list);
	
	if (dir_list->focus_row >= 1) {
		gtk_clist_row_move (dir_list, dir_list->focus_row, dir_list->focus_row-1);
		update_move_buttons (dialog, dir_list->focus_row);
	}
}


static void on_dir_move_down (GtkButton *button, GnomeCmdBookmarkDialog *dialog)
{
	GtkCList *dir_list = GTK_CLIST (dialog->priv->dir_list);

	if (dir_list->focus_row >= 0) {
		gtk_clist_row_move (dir_list, dir_list->focus_row, dir_list->focus_row+1);
		update_move_buttons (dialog, dir_list->focus_row);
	}
}


static void
on_dir_moved (GtkCList *clist, gint arg1, gint arg2,
			  GnomeCmdBookmarkDialog *dialog)
{
	GList *bookmarks = dialog->priv->sel_group->bookmarks;
	gpointer data;

	if (!bookmarks
		|| MAX (arg1, arg2) >= g_list_length (bookmarks)
		|| MIN (arg1, arg2) < 0
		|| arg1 == arg2)
		return;

	data = g_list_nth_data (bookmarks, arg1);
	bookmarks = g_list_remove (bookmarks, data);
	
	bookmarks = g_list_insert (bookmarks, data, arg2);

	dialog->priv->sel_group->bookmarks = bookmarks;
}


static void on_close (GtkButton *button, GnomeCmdBookmarkDialog *dialog)
{
	gtk_widget_destroy (GTK_WIDGET (dialog));
}


static gboolean
on_dialog_keypress (GnomeCmdBookmarkDialog *dialog, GdkEventKey *event)
{
	if (event->keyval == GDK_Escape) {
		gtk_widget_hide (GTK_WIDGET (dialog));
		return TRUE;
	}
	else if (event->keyval == GDK_Return) {
		on_dir_goto (NULL, dialog);
		return TRUE;
	}

	return FALSE;
}


static void on_dir_selected (GtkCList *list, gint row, gint column,
							  GdkEventButton *event, GnomeCmdBookmarkDialog *dialog)
{
	if (event && event->type == GDK_2BUTTON_PRESS) {
		on_dir_goto (NULL, dialog);
	}
	else {
		dialog->priv->sel_bookmark = gtk_clist_get_row_data (list, row);
		gtk_widget_set_sensitive (dialog->priv->remove_btn, TRUE);
		gtk_widget_set_sensitive (dialog->priv->edit_btn, TRUE);
		gtk_widget_set_sensitive (dialog->priv->goto_btn, TRUE);
	}

	update_move_buttons (dialog, row);
}


static void on_dir_unselected (GtkCList *list, gint row, gint column,
							   GdkEventButton *event, GnomeCmdBookmarkDialog *dialog)
{
	gtk_widget_set_sensitive (dialog->priv->remove_btn, FALSE);
	gtk_widget_set_sensitive (dialog->priv->edit_btn, FALSE);
	gtk_widget_set_sensitive (dialog->priv->goto_btn, FALSE);
	gtk_widget_set_sensitive (dialog->priv->move_up_btn, FALSE);
	gtk_widget_set_sensitive (dialog->priv->move_down_btn, FALSE);
}


static void
on_scroll_vertical                  (GtkCList        *clist,
									 GtkScrollType    scroll_type,
									 gfloat           position,
									 gpointer         data)
{
	gtk_clist_select_row (clist, clist->focus_row, 0);
}


static void
add_groups (GnomeCmdBookmarkDialog *dialog)
{
	gchar *text[3];
	gint row;
	GList *groups = dialog->priv->groups;

	while (groups) {
		GnomeCmdBookmarkGroup *group = (GnomeCmdBookmarkGroup*)groups->data;
	
		text[0] = NULL;
		text[1] = group->alias;
		text[2] = NULL;
		
		row = gnome_cmd_combo_append (
			GNOME_CMD_COMBO (dialog->priv->combo), text, group);
		gnome_cmd_combo_set_pixmap (
			GNOME_CMD_COMBO (dialog->priv->combo), row, 0,
			IMAGE_get_gnome_cmd_pixmap (group->server?PIXMAP_SERVER_SMALL:PIXMAP_HOME));

		groups = groups->next;
	}
}


static void
add_bookmarks (GnomeCmdBookmarkDialog *dialog)
{
	GList *servers;
	GnomeCmdBookmarkGroup *group, *default_group;
	GnomeCmdConnection *con = gnome_cmd_file_selector_get_connection (
		gnome_cmd_main_win_get_active_fs (main_win));

	g_return_if_fail (con != NULL);
	
	/* First add local bookmarks */
	default_group = group = gnome_cmd_data_get_local_bookmarks ();
	group->data = (gpointer)dialog;
	dialog->priv->groups = g_list_append (dialog->priv->groups, group);	

	/* Then add bookmarks for the ftp-servers */
	servers = gnome_cmd_data_get_ftp_servers ();
	while (servers) {
		GnomeCmdFtpServer *server = (GnomeCmdFtpServer*)servers->data;
		group = gnome_cmd_ftp_server_get_bookmarks (server);
		group->data = (gpointer)dialog;
		dialog->priv->groups = g_list_append (dialog->priv->groups, group);

		if (con->server == group->server)
			default_group = group;
		
		servers = servers->next;
	}

	add_groups (dialog);
	set_selected_group (dialog, default_group);
}


static void
on_group_combo_item_selected (GnomeCmdCombo *group_combo,
							  GnomeCmdBookmarkGroup *group,
							  GnomeCmdBookmarkDialog *dialog)
{
	g_return_if_fail (group != NULL);

	set_selected_group (dialog, group);
}


static void
on_column_resize (GtkCList *clist, gint column, gint width,
				  GnomeCmdBookmarkDialog *dialog)
{
	gnome_cmd_data_set_bookmark_dialog_col_width (column, width);
}


/*******************************
 * Gtk class implementation
 *******************************/

static void
destroy (GtkObject *object)
{
	GnomeCmdBookmarkDialog *dialog = GNOME_CMD_BOOKMARK_DIALOG (object);

	g_list_free (dialog->priv->groups);
	g_free (dialog->priv);

	gnome_cmd_main_win_update_bookmarks (main_win);
	
	if (GTK_OBJECT_CLASS (parent_class)->destroy)
		(*GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}


static void
map (GtkWidget *widget)
{
	if (GTK_WIDGET_CLASS (parent_class)->map != NULL)
		GTK_WIDGET_CLASS (parent_class)->map (widget);
}


static void
class_init (GnomeCmdBookmarkDialogClass *class)
{
	GtkObjectClass *object_class;
	GtkWidgetClass *widget_class;

	object_class = GTK_OBJECT_CLASS (class);
	widget_class = GTK_WIDGET_CLASS (class);

	parent_class = gtk_type_class (gnome_dialog_get_type ());
	object_class->destroy = destroy;
	widget_class->map = map;
}


static void
init (GnomeCmdBookmarkDialog *in_dialog)
{
	GtkWidget *dialog;
	GtkWidget *vbox;
	GtkWidget *vsep;
	GtkWidget *hbox, *group_hbox;
	GtkWidget *bbox;
	GtkWidget *dir_list;
	GtkWidget *button;
	GtkWidget *fixed;
	GtkWidget *label;
	GtkAccelGroup *accel_group;

	in_dialog->priv = g_new (GnomeCmdBookmarkDialogPrivate, 1);
	in_dialog->priv->groups = NULL;
	
	accel_group = gtk_accel_group_new ();

	dialog = GTK_WIDGET (in_dialog);
	gtk_object_set_data (GTK_OBJECT (dialog), "dialog", dialog);
	gtk_window_set_default_size (GTK_WINDOW (dialog), 500, 400);
	gtk_window_set_policy (GTK_WINDOW (dialog), FALSE, TRUE, FALSE);
	gtk_window_set_title (GTK_WINDOW (dialog), _("Bookmarks"));

	vbox = GNOME_DIALOG (dialog)->vbox;
	gtk_object_set_data (GTK_OBJECT (dialog), "vbox", vbox);
	gtk_widget_show (vbox);
	gtk_box_set_spacing (GTK_BOX (vbox), 6);

	hbox = create_hbox (dialog, FALSE, 6);
	gtk_box_pack_start (GTK_BOX (vbox), hbox, TRUE, TRUE, 0);


	
	vbox = create_vbox (dialog, FALSE, 6);
	gtk_box_pack_start (GTK_BOX (hbox), vbox, TRUE, TRUE, 0);

	group_hbox = create_hbox (dialog, FALSE, 6);
	gtk_box_pack_start (GTK_BOX (vbox), group_hbox, FALSE, TRUE, 0);

	label = create_label (dialog, _("Bookmark Group: "));
	gtk_box_pack_start (GTK_BOX (group_hbox), label, FALSE, TRUE, 0);

	in_dialog->priv->combo = create_clist_combo (dialog, 2, NULL);
	gtk_clist_set_column_width (
		GTK_CLIST (GNOME_CMD_COMBO (in_dialog->priv->combo)->list), 0, 20);	
	gtk_clist_set_column_width (
		GTK_CLIST (GNOME_CMD_COMBO (in_dialog->priv->combo)->list), 1, 60);
	gtk_box_pack_start (GTK_BOX (group_hbox), in_dialog->priv->combo, TRUE, TRUE, 0);
	
	dir_list = create_clist (dialog, "dir_list", 3, 16,
							 on_dir_selected, on_dir_moved);
	gtk_box_pack_start (GTK_BOX (vbox), dir_list, TRUE, TRUE, 3);
	create_clist_column (dir_list, 0,
						 gnome_cmd_data_get_bookmark_dialog_col_width (0), "");
	create_clist_column (dir_list, 1,
						 gnome_cmd_data_get_bookmark_dialog_col_width (1), _("name"));
	create_clist_column (dir_list, 2,
						 gnome_cmd_data_get_bookmark_dialog_col_width (2), _("path"));

	in_dialog->priv->dir_list = lookup_widget (GTK_WIDGET (dialog), "dir_list");
	


	vsep = create_vsep (dialog);
	gtk_box_pack_start (GTK_BOX (hbox), vsep, FALSE, FALSE, 0);
	
	bbox = create_vbox (dialog, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (hbox), bbox, FALSE, FALSE, 0);
	
	in_dialog->priv->goto_btn = create_named_button (dialog, _("_Goto"), "goto_button", on_dir_goto);
	GTK_WIDGET_SET_FLAGS (in_dialog->priv->goto_btn, GTK_CAN_DEFAULT);
	gtk_box_pack_start (GTK_BOX (bbox), in_dialog->priv->goto_btn, FALSE, TRUE, 0);
	gtk_widget_set_sensitive (GTK_WIDGET (in_dialog->priv->goto_btn), FALSE);

	in_dialog->priv->edit_btn = create_named_button (dialog, _("_Edit"), "edit_button", on_dir_edit);
	GTK_WIDGET_SET_FLAGS (in_dialog->priv->edit_btn, GTK_CAN_DEFAULT);
	gtk_box_pack_start (GTK_BOX (bbox), in_dialog->priv->edit_btn, FALSE, TRUE, 0);
	gtk_widget_set_sensitive (GTK_WIDGET (in_dialog->priv->edit_btn), FALSE);
	
	in_dialog->priv->remove_btn = create_named_button (dialog, _("_Remove"), "remove_button", on_dir_remove);
	GTK_WIDGET_SET_FLAGS (in_dialog->priv->remove_btn, GTK_CAN_DEFAULT);
	gtk_box_pack_start (GTK_BOX (bbox), in_dialog->priv->remove_btn, FALSE, TRUE, 0);
	gtk_widget_set_sensitive (GTK_WIDGET (in_dialog->priv->remove_btn), FALSE);
	
	in_dialog->priv->move_up_btn = create_named_stock_button (
		dialog, GNOME_STOCK_BUTTON_UP, "move_up_button",
		GTK_SIGNAL_FUNC (on_dir_move_up));
	GTK_WIDGET_SET_FLAGS (in_dialog->priv->move_up_btn, GTK_CAN_DEFAULT);
	gtk_box_pack_start (GTK_BOX (bbox), in_dialog->priv->move_up_btn, FALSE, TRUE, 0);
	gtk_widget_set_sensitive (GTK_WIDGET (in_dialog->priv->move_up_btn), FALSE);

	in_dialog->priv->move_down_btn = create_named_stock_button (
		dialog, GNOME_STOCK_BUTTON_DOWN, "move_down_button",
		GTK_SIGNAL_FUNC (on_dir_move_down));
	GTK_WIDGET_SET_FLAGS (in_dialog->priv->move_down_btn, GTK_CAN_DEFAULT);
	gtk_box_pack_start (GTK_BOX (bbox), in_dialog->priv->move_down_btn, FALSE, TRUE, 0);
	gtk_widget_set_sensitive (GTK_WIDGET (in_dialog->priv->move_down_btn), FALSE);

	fixed = gtk_fixed_new ();
	gtk_widget_ref (fixed);
	gtk_object_set_data_full (GTK_OBJECT (dialog), "fixed", fixed,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (fixed);
	gtk_box_pack_start (GTK_BOX (bbox), fixed, TRUE, TRUE, 0);

	button = create_stock_button (
		dialog, GNOME_STOCK_BUTTON_CLOSE, GTK_SIGNAL_FUNC (on_close));
	GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
	gtk_box_pack_start (GTK_BOX (bbox), button, FALSE, TRUE, 0);


	gtk_signal_connect (GTK_OBJECT (dialog), "key-press-event",
						GTK_SIGNAL_FUNC (on_dialog_keypress), dialog);
	gtk_signal_connect_after (GTK_OBJECT (in_dialog->priv->dir_list), "scroll_vertical",
							  GTK_SIGNAL_FUNC (on_scroll_vertical), NULL);
	gtk_signal_connect (GTK_OBJECT (in_dialog->priv->dir_list), "unselect-row",
						GTK_SIGNAL_FUNC (on_dir_unselected), dialog);
	gtk_signal_connect (GTK_OBJECT (in_dialog->priv->dir_list), "resize-column",
						GTK_SIGNAL_FUNC (on_column_resize), dialog);
	gtk_signal_connect (GTK_OBJECT (in_dialog->priv->combo), "item-selected",
						GTK_SIGNAL_FUNC (on_group_combo_item_selected), dialog);

	gtk_widget_grab_focus (in_dialog->priv->dir_list);
}


/***********************************
 * Public functions
 ***********************************/

GtkWidget*
gnome_cmd_bookmark_dialog_new (void)
{
	GnomeCmdBookmarkDialog *dialog = gtk_type_new (gnome_cmd_bookmark_dialog_get_type ());

	add_bookmarks (dialog);
	
	return GTK_WIDGET (dialog);
}



GtkType
gnome_cmd_bookmark_dialog_get_type (void)
{
	static GtkType dlg_type = 0;

	if (dlg_type == 0)
	{
		GtkTypeInfo dlg_info =
		{
			"GnomeCmdBookmarkDialog",
			sizeof (GnomeCmdBookmarkDialog),
			sizeof (GnomeCmdBookmarkDialogClass),
			(GtkClassInitFunc) class_init,
			(GtkObjectInitFunc) init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL
		};

		dlg_type = gtk_type_unique (gnome_dialog_get_type (), &dlg_info);
	}
	return dlg_type;
}


static gboolean
on_new_bookmark_ok (GnomeCmdStringDialog *string_dialog, const gchar **values, gpointer data)
{
	GnomeCmdBookmark *bookmark = g_new (GnomeCmdBookmark, 1);
	GnomeCmdConnection *con = gnome_cmd_file_selector_get_connection (
		gnome_cmd_main_win_get_active_fs (main_win));
	GnomeCmdBookmarkGroup *group;
	
	if (con->server)
		group = gnome_cmd_ftp_server_get_bookmarks (con->server);
	else
		group = gnome_cmd_data_get_local_bookmarks ();

	bookmark->name = g_strdup (values[0]);
	bookmark->path = g_strdup (values[1]);
	bookmark->group = group;
	
	group->bookmarks = g_list_append (group->bookmarks, bookmark);

	gnome_cmd_main_win_update_bookmarks (main_win);
	
	return TRUE;
}


void
gnome_cmd_bookmark_add_current (void)
{
	GnomeCmdDir *cwd = gnome_cmd_file_selector_get_directory (
		gnome_cmd_main_win_get_active_fs (main_win));
	const gchar *path = gnome_cmd_dir_get_path (cwd);
	
	show_bookmark_dialog (g_basename (path),
						  path,
						  _("New Bookmark"),
						  (GnomeCmdStringDialogCallback)on_new_bookmark_ok,
						  NULL);
}


void
gnome_cmd_bookmark_goto (GnomeCmdBookmark *bookmark)
{
	gboolean using_old;
	GnomeCmdConnection *con, *current_con;
	GnomeCmdFileSelector *fs = gnome_cmd_main_win_get_active_fs (main_win);
	g_return_if_fail (GNOME_CMD_IS_FILE_SELECTOR (fs));

	current_con = gnome_cmd_file_selector_get_connection (fs);
	con = get_connection_by_bookmark (bookmark, &using_old);
	g_return_if_fail (con != NULL);

	if (current_con == con)
		gnome_cmd_file_selector_goto_directory (fs, bookmark->path);
	else {
		if (using_old) {
			GnomeCmdDir *dir = gnome_cmd_dir_new_with_values (
				con->base_uri, bookmark->path, TRUE);
			gnome_cmd_file_selector_set_connection (fs, con, dir);
		}
		else {
			if (gnome_cmd_file_selector_set_connection (fs, con, NULL))
				gnome_cmd_connection_add (con);
			else
				gnome_cmd_connection_free (con);
		}
	}
}



