#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "fft.h"

/* ---------------------------------------------------------------------- */

static unsigned int rbits32(unsigned int w)
{
        w = ((w >>  1) & 0x55555555) | ((w <<  1) & 0xaaaaaaaa);
        w = ((w >>  2) & 0x33333333) | ((w <<  2) & 0xcccccccc);
        w = ((w >>  4) & 0x0f0f0f0f) | ((w <<  4) & 0xf0f0f0f0);
        w = ((w >>  8) & 0x00ff00ff) | ((w <<  8) & 0xff00ff00);
        w = ((w >> 16) & 0x0000ffff) | ((w << 16) & 0xffff0000);
        return w;
}

static double hamming(double x)
{
	return 0.54 - 0.46 * cos(2 * M_PI * x);
}

/* ---------------------------------------------------------------------- */

struct fft *init_fft(int len)
{
	struct fft *f;
	int i;

	if ((f = calloc(1, sizeof(struct fft))) == NULL)
		return NULL;

	f->twiddles = calloc(len, sizeof(complex));
	f->bitrev = calloc(len, sizeof(unsigned int));
	f->window = calloc(len, sizeof(float));

	if (!f->twiddles || !f->bitrev || !f->window) {
		free(f->twiddles);
		free(f->bitrev);
		free(f->window);
		free(f);
		return NULL;
	}

	f->fftlen = len;

	f->fftlenlog = 0;
	while (len) {
		f->fftlenlog++;
		len >>= 1;
	}
	f->fftlenlog--;

	for (i = 0; i < f->fftlen; i++) {
		f->twiddles[i].re = cos((double) i / f->fftlen * 2.0 * M_PI);
		f->twiddles[i].im = sin((double) i / f->fftlen * 2.0 * M_PI);
		f->bitrev[i] = rbits32(i) >> (32 - f->fftlenlog);
		f->window[i] = hamming(i / (f->fftlen - 1.0));
	}

	return f;
}

/*
 * Radix-2 decimation-in-time FFT routine.
 */
void fft(struct fft *f, complex *in, complex *out)
{
	int i, j, k;
	int s, sep, width, top, bot;
	float tr, ti;

	/*
	 * Order the samples in bit reverse order.
	 */
	for (i = 0; i < f->fftlen; i++)
		out[f->bitrev[i]] = in[i];

	/*
	 * In-place FFT.
	 */
	for (sep = s = 1; s <= f->fftlenlog; s++) {
		width = sep;	/* butterfly width      = 2^(s-1)  */
		sep <<= 1;	/* butterfly separation = 2^s      */

		for (j = 0; j < width; j++) {
			k = f->fftlen * j / sep;

			for (top = j; top < f->fftlen; top += sep) {
				bot = top + width;

				tr = out[bot].re * f->twiddles[k].re +
				     out[bot].im * f->twiddles[k].im;

				ti = out[bot].im * f->twiddles[k].re -
				     out[bot].re * f->twiddles[k].im;

				out[bot].re = out[top].re - tr;
				out[bot].im = out[top].im - ti;

				out[top].re = out[top].re + tr;
				out[top].im = out[top].im + ti;
			}
		}
	}
}

/*
 * Radix-2 decimation-in-time FFT routine
 * (with real input and a window function).
 */
void fft2(struct fft *f, float *in, complex *out)
{
	int i, j, k;
	int s, sep, width, top, bot;
	float tr, ti;

	/*
	 * Order the samples in bit reverse order and apply window.
	 */
	for (i = 0; i < f->fftlen; i++) {
		out[f->bitrev[i]].re = in[i] * f->window[i];
		out[f->bitrev[i]].im = 0.0;
	}

	/*
	 * In-place FFT.
	 */
	for (sep = s = 1; s <= f->fftlenlog; s++) {
		width = sep;	/* butterfly width      = 2^(s-1)  */
		sep <<= 1;	/* butterfly separation = 2^s      */

		for (j = 0; j < width; j++) {
			k = f->fftlen * j / sep;

			for (top = j; top < f->fftlen; top += sep) {
				bot = top + width;

				tr = out[bot].re * f->twiddles[k].re +
				     out[bot].im * f->twiddles[k].im;

				ti = out[bot].im * f->twiddles[k].re -
				     out[bot].re * f->twiddles[k].im;

				out[bot].re = out[top].re - tr;
				out[bot].im = out[top].im - ti;

				out[top].re = out[top].re + tr;
				out[top].im = out[top].im + ti;
			}
		}
	}
}

/* ---------------------------------------------------------------------- */

struct slfft *init_slfft(int len, int first, int last)
{
	struct slfft *s;
	int i;

	if ((s = calloc(1, sizeof(struct slfft))) == NULL)
		return NULL;

	s->twiddles = calloc(len, sizeof(complex));
	s->history = calloc(len, sizeof(float));
	s->bins = calloc(len, sizeof(complex));

	if (!s->twiddles || !s->history || !s->bins) {
		free(s->twiddles);
		free(s->history);
		free(s->bins);
		free(s);
		return NULL;
	}

	s->fftlen = len;
	s->first = first;
	s->last = last;

	for (i = 0; i < len; i++) {
		s->twiddles[i].re = cos(i * 2.0 * M_PI / len);
		s->twiddles[i].im = sin(i * 2.0 * M_PI / len);
	}
	memset(s->history, 0, sizeof(s->history));
	memset(s->bins, 0, sizeof(s->bins));

	return s;
}

/*
 * Sliding FFT, real input, complex output
 */
complex *slfft(struct slfft *s, float new)
{
	float ti, tr, old;
	int i;

	/* restore the sample fftlen samples back */
	old = s->history[s->ptr];

	/* save the new sample */
	s->history[s->ptr] = new;

	/* advance the history pointer */
	s->ptr = (s->ptr + 1) % s->fftlen;

	/* calculate the wanted bins */
	for (i = s->first; i < s->last; i++) {
		tr = s->bins[i].re - old + new;
		ti = s->bins[i].im;

		s->bins[i].re = (tr * s->twiddles[i].re) -
			        (ti * s->twiddles[i].im);

		s->bins[i].im = (tr * s->twiddles[i].im) +
			        (ti * s->twiddles[i].re);
	}

	return s->bins;
}

/* ---------------------------------------------------------------------- */

