/*
 * Copyright 1999-2006 University of Chicago
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "./library/globus_rls_client.h"

#include <signal.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <ctype.h>
#include "version.h"

static globus_module_descriptor_t	*modules[] = {
  GLOBUS_COMMON_MODULE,
  GLOBUS_IO_MODULE,
  GLOBUS_RLS_CLIENT_MODULE,
};					      
#define NMODS	(sizeof(modules) / sizeof(globus_module_descriptor_t *))

static globus_result_t	doadd(int argc, char **argv);
static globus_result_t	doattradd(int argc, char **argv);
static globus_result_t	doattrdelete(int argc, char **argv);
static globus_result_t	doattrbulkadd(int argc, char **argv);
static globus_result_t	doattrbulkdelete(int argc, char **argv);
static globus_result_t	doattrbulkquery(int argc, char **argv);
static globus_result_t	doattrmodify(int argc, char **argv);
static globus_result_t	doattrsearch(int argc, char **argv);
static globus_result_t	dobulkadd(int argc, char **argv);
static globus_result_t	dobulkcreate(int argc, char **argv);
static globus_result_t	dobulkdelete(int argc, char **argv);
static globus_result_t	dobulkexistslrclfn(int argc, char **argv);
static globus_result_t	dobulkexistslrcpfn(int argc, char **argv);
static globus_result_t	dobulkexistsrlilfn(int argc, char **argv);
static globus_result_t	dobulkquerylrclfn(int argc, char **argv);
static globus_result_t	dobulkquerylrcpfn(int argc, char **argv);
static globus_result_t	dobulkqueryrlilfn(int argc, char **argv);
static globus_result_t	dobulkrenamelfn(int argc, char **argv);
static globus_result_t	dobulkrenamepfn(int argc, char **argv);
static globus_result_t	doclear(int argc, char **argv);
static globus_result_t	docreate(int argc, char **argv);
static globus_result_t	dodelete(int argc, char **argv);
static globus_result_t	dodefine(int argc, char **argv);
static globus_result_t	doexit(int argc, char **argv);
static globus_result_t	dohelp(int argc, char **argv);
static globus_result_t	doqueryattr(int argc, char **argv);
static globus_result_t	doquerylrclfn(int argc, char **argv);
static globus_result_t	doquerylrcpfn(int argc, char **argv);
static globus_result_t	doqueryrlilfn(int argc, char **argv);
static globus_result_t	doquerywclrclfn(int argc, char **argv);
static globus_result_t	doquerywclrcpfn(int argc, char **argv);
static globus_result_t	doquerywcrlilfn(int argc, char **argv);
static globus_result_t	dosetclearvalues(int argc, char **argv);
static globus_result_t	dosetreslimit(int argc, char **argv);
static globus_result_t	dosettimeout(int argc, char **argv);
static globus_result_t	doshowattr(int argc, char **argv);
static globus_result_t	doundefine(int argc, char **argv);
static globus_result_t  doexistslrc(int argc, char **argv);
static globus_result_t  doexistsrlilfn(int argc, char **argv);
static globus_result_t	dorenamelfn(int argc, char **argv);
static globus_result_t	dorenamepfn(int argc, char **argv);

typedef enum show_ {
  rls_cli_show_all,
  rls_cli_show_exists,
  rls_cli_show_nexists
} SHOW;

typedef struct kwd_ {
  char			*kwd;
  struct kwd_		*nextkwd;
  globus_result_t	(*f)(int argc, char **argv);
  int			narg;
  char			*arghelp;
} KWD;

static KWD bulkexistslrckwds[] = {
  { "lfn", NULL, dobulkexistslrclfn, -1, "<val> <lfns...>" },
  { "pfn", NULL, dobulkexistslrcpfn, -1, "<val> <pfns...>" },
  { NULL, NULL, NULL, 0, NULL }
};

static KWD bulkexistsrlikwds[] = {
  { "lfn", NULL, dobulkexistsrlilfn, -1, "<val> <lfns...>" },
  { NULL, NULL, NULL, 0, NULL }
};

static KWD bulkexistskwds[] = {
  { "lrc", bulkexistslrckwds, NULL, 0, NULL },
  { "rli", bulkexistsrlikwds, NULL, 0, NULL },
  { NULL, NULL, NULL, 0, NULL }
};

static KWD bulkquerylrckwds[] = {
  { "lfn", NULL, dobulkquerylrclfn, -1, "<lfns...>" },
  { "pfn", NULL, dobulkquerylrcpfn, -1, "<pfns...>" },
  { NULL, NULL, NULL, 0, NULL }
};

static KWD bulkqueryrlikwds[] = {
  { "lfn", NULL, dobulkqueryrlilfn, -1, "<lfns...>" },
  { NULL, NULL, NULL, 0, NULL }
};

static KWD bulkquerykwds[] = {
  { "lrc", bulkquerylrckwds, NULL, 0, NULL },
  { "rli", bulkqueryrlikwds, NULL, 0, NULL },
  { NULL, NULL, NULL, 0, NULL }
};

static KWD bulkrenamekwds[] = {
  { "lfn", NULL, dobulkrenamelfn, -1, "<lfn> <lfn> ..." },
  { "pfn", NULL, dobulkrenamepfn, -1, "<pfn> <pfn> ..." },
  { NULL, NULL, NULL, 0, NULL }
};

static KWD bulkkwds[] = {
  { "add", NULL, dobulkadd, -1, "<lfn> <pfn> ..." },
  { "create", NULL, dobulkcreate, -1, "<lfn> <pfn> ..." },
  { "delete", NULL, dobulkdelete, -1, "<lfn> <pfn> ..." },
  { "exists", bulkexistskwds, NULL, 0, NULL },
  { "query", bulkquerykwds, NULL, 0, NULL },
  { "rename", bulkrenamekwds, NULL, 0, NULL },
  { NULL, NULL, NULL, 0, NULL }
};

static KWD querylrckwds[] = {
  { "lfn", NULL, doquerylrclfn, 1, "<lfn>" },
  { "pfn", NULL, doquerylrcpfn, 1, "<pfn>" },
  { NULL, NULL, NULL, 0, NULL }
};

static KWD queryrlikwds[] = {
  { "lfn", NULL, doqueryrlilfn, 1, "<lfn>" },
  { NULL, NULL, NULL, 0, NULL }
};

static KWD querywclrckwds[] = {
  { "lfn", NULL, doquerywclrclfn, 1, "<lfn-pattern>" },
  { "pfn", NULL, doquerywclrcpfn, 1, "<pfn-pattern>" },
  { NULL, NULL, NULL, 0, NULL }
};

static KWD querywcrlikwds[] = {
  { "lfn", NULL, doquerywcrlilfn, 1, "<lfn-pattern>" },
  { NULL, NULL, NULL, 0, NULL }
};

static KWD	querywckwds[] = {
  { "lrc", querywclrckwds, NULL, 0, NULL },
  { "rli", querywcrlikwds, NULL, 0, NULL },
  { NULL, NULL, NULL, 0, NULL }
};

static KWD	querykwds[] = {
  { "lrc", querylrckwds, NULL, 0, NULL },
  { "rli", queryrlikwds, NULL, 0, NULL },
  { "wildcard", querywckwds, NULL, 0, NULL },
  { NULL, NULL, NULL, 0, NULL }
};

static KWD renamekwds[] = {
  { "lfn", NULL, dorenamelfn, 2, "<lfn> <lfn>" },
  { "pfn", NULL, dorenamepfn, 2, "<pfn> <pfn>" },
  { NULL, NULL, NULL, 0, NULL }
};

static KWD	existsrlikwds[] = {
  { "lfn", NULL, doexistsrlilfn, 1, "<lfn>" },
  { NULL, NULL, NULL, 0, NULL }
};

static KWD	existskwds[] = {
  { "lrc", NULL, doexistslrc, 2, "<obj type> <val>" },
  { "rli", existsrlikwds, NULL, 0, NULL },
  { NULL, NULL, NULL, 0, NULL }
};

static KWD	attrbulkkwds[] = {
  { "add", NULL, doattrbulkadd, -1, "<object> <attr> <obj type> <attr type> <val> ..."},
  { "delete", NULL, doattrbulkdelete, -1, "<object> <attr> <obj type> <attr type>..." },
  { "query", NULL, doattrbulkquery, -1, "<attr> <obj type> <object> ..." },
  { NULL, NULL, NULL, 0, NULL }
};

static KWD	attrkwds[] = {
  { "add", NULL, doattradd, 5, "<object> <attr> <obj type> <attr type> <val>"},
  { "bulk", attrbulkkwds, NULL, 0, NULL },
  { "define", NULL, dodefine, 3, "<attr> <obj type> <attr type>" },
  { "delete", NULL, doattrdelete, 3, "<object> <attr> <obj type>" },
  { "modify", NULL, doattrmodify, 5, "<object> <attr> <obj type> <attr type> <val>" },
  { "query", NULL, doqueryattr, 3, "<object> <attr> <obj type>" },
  { "search",NULL,doattrsearch, 5, "<attr> <obj type> <op> <attr type> <val>"},
  { "show", NULL, doshowattr, 2, "<attr> <obj type>" },
  { "undefine", NULL, doundefine, 2, "<attr> <obj type>" },
  { NULL, NULL, NULL, 0, NULL }
};

static KWD	setkwds[] = {
  { "clearvalues", NULL, dosetclearvalues, 1, "true|false" },
  { "reslimit", NULL, dosetreslimit, 1, "reslimit" },
  { "timeout", NULL, dosettimeout, 1, "timeout" },
  { NULL, NULL, NULL, 0, NULL }
};

static KWD	cmdkwds[] = {
  { "add", NULL, doadd, 2, "<lfn> <pfn>"},
  { "attribute", attrkwds, NULL, 0, NULL },
  { "bulk", bulkkwds, NULL, 0, NULL },
  { "clear", NULL, doclear, 0, NULL },
  { "create", NULL, docreate, 2, "<lfn> <pfn>" },
  { "delete", NULL, dodelete, 2, "<lfn> <pfn>" },
  { "exists", existskwds, NULL, 0, NULL },
  { "exit", NULL, doexit, 0, NULL },
  { "help", NULL, dohelp, 0, NULL },
  { "query", querykwds, NULL, 0, NULL },
  { "rename", renamekwds, NULL, 0, NULL },
  { "set", setkwds, NULL, 0, NULL },
  { NULL, NULL, NULL, 0, NULL }
};

static KWD	attrtypekwds[] = {
  { "date", NULL, NULL, (int) globus_rls_attr_type_date, "" },
  { "float", NULL, NULL, (int) globus_rls_attr_type_flt, "" },
  { "integer", NULL, NULL, (int) globus_rls_attr_type_int, "" },
  { "string", NULL, NULL, (int) globus_rls_attr_type_str, "" },
  { NULL, NULL, NULL, 0, NULL }
};

static KWD	showexistskwds[] = {
  { "showall", NULL, NULL, (int) rls_cli_show_all, "" },
  { "showexists", NULL, NULL, (int) rls_cli_show_exists, "" },
  { "shownotexists", NULL, NULL, (int) rls_cli_show_nexists, "" },
  { NULL, NULL, NULL, 0, NULL }
};

static globus_result_t		pcmd(int argc, char **argv);
static void			lex(char *buf, int *argc, char **argv);
static int			findkwd(char *kwd, KWD *kwdtab);
static void			showbulkfailures(globus_list_t *l);
static void			showexists(char *objtype, char *val);
static void			showexistsbulk(globus_list_t *l, SHOW show);
static void			showresult(globus_list_t *l);
static void			showresultbulk(globus_list_t *l);
static void			showattr(globus_list_t *l);
static void			showattrdef(globus_list_t *l);
static void			showattrobj(globus_list_t *l);
static int			a2obj(char *s, globus_rls_obj_type_t *ot);
static int			a2attr(char *s, globus_rls_attr_type_t *at);
static int			a2op(char *s, globus_rls_attr_op_t *op);
static char			*attr2a(globus_rls_attr_type_t type);
static void			hsig(int s);
static void			cleanexit(int s);
static void			showcmds(int argc, char **argv, KWD *k);
static void			freelistdatums(globus_list_t *l);
static void			usage(char *prog);
static globus_rls_handle_t	*h = NULL;
static globus_rls_pattern_t	pattype = rls_pattern_unix;
static int			quitting = 0;
static globus_bool_t		clearvalues = GLOBUS_FALSE;

#define MAXCMD	100

static char				*cmds[MAXCMD];
static int				cmdi;
static int				reslimit = 0;
static int				timeout = 0;
static FILE				*infile=NULL;
static globus_bool_t	decorate = GLOBUS_FALSE;

int
main(int argc, char **argv)

{
  int			i;
  globus_result_t	r;
  int			rc;
  char			*rlsserver;
  char			errmsg[MAXERRMSG];
  int			interactive;
  extern char		*optarg;
  extern int		optind;
  char			buf[BUFSIZ];
  int			iargc;
  char			*iargv[MAXCMD];
  struct        sigaction	sa;

  sigemptyset(&sa.sa_mask);
  sa.sa_handler = hsig;
  sa.sa_flags = SA_RESTART;

  sigaction(SIGINT, &sa, NULL);
  sigaction(SIGTERM, &sa, NULL); 

  for (i = 0; i < NMODS; i++)
    if ((rc = globus_module_activate(modules[i])) != GLOBUS_SUCCESS) {
      fprintf(stderr, "globus_module_activate(%d): %d\n", i, rc);
      exit(1);
    }

  while ((i = getopt(argc, argv, "cdhi:l:st:uv")) != -1)
    switch (i) {
      case 'c': clearvalues = GLOBUS_TRUE;
		break;
      case 'd': decorate = GLOBUS_TRUE;
		break;
      case 'h': usage(*argv);
		break;
      case 'l': reslimit = atoi(optarg);
		break;
      case 's':	pattype = rls_pattern_sql;
		break;
      case 't': timeout = atoi(optarg);
		break;
      case 'u': pattype = rls_pattern_unix;
		break;
      case 'v':	printf("Version: %d.%d\n", MAJOR, MINOR);
		exit(0);
      case 'i': if (!strcmp(optarg, "-")) {
                    infile = stdin;
                }
                else if ( !(infile=fopen(optarg,"r")) ) {
                    perror(optarg);
                    cleanexit(1);
                }
                break;
      case '?': usage(*argv);
    }
  if (optind >= argc)
    usage(*argv);
  if (optind >= argc - 1) {
    if (infile)
      usage(*argv);
    interactive = 1;
  }
  else
    interactive = 0;
  rlsserver = argv[argc-1];

  if ((r = globus_rls_client_connect(rlsserver, &h)) != GLOBUS_SUCCESS) {
    globus_rls_client_error_info(r, NULL, errmsg, MAXERRMSG, GLOBUS_FALSE);
    fprintf(stderr, "connect(%s): %s\n", rlsserver, errmsg);
    cleanexit(1);
  }
  if (timeout)
    globus_rls_client_set_timeout(timeout);

#ifdef foo
{globus_rls_attribute_object_t ao, ao2;
 globus_list_t *al = NULL;
 globus_list_t *rl = NULL;
 globus_result_t r;

globus_list_insert(&al, "y");
globus_list_insert(&al, "zz");
r = globus_rls_client_lrc_attr_value_get_bulk(h, al, "iattr", globus_rls_obj_lrc_lfn,
					  &rl);
if (r != GLOBUS_SUCCESS) {
  globus_rls_client_error_info(r, NULL, errmsg, MAXERRMSG, GLOBUS_FALSE);
  fprintf(stderr, "vgb: %s\n", errmsg);
} else
  showattrobj(rl);
cleanexit(0);

 ao.key = "x";
 ao.attr.name = "iattr";
 ao.attr.objtype = globus_rls_obj_lrc_lfn;
 ao.attr.type = globus_rls_attr_type_int;
 ao.attr.val.i = 31;
 globus_list_insert(&al, &ao);
 ao2.key = "xx";
 ao2.attr.name = "iattr";
 ao2.attr.objtype = globus_rls_obj_lrc_lfn;
 ao2.attr.type = globus_rls_attr_type_int;
 ao2.attr.val.i = -211;
 globus_list_insert(&al, &ao2);
r = globus_rls_client_lrc_attr_add_bulk(h, al, &rl);
if (r != GLOBUS_SUCCESS) {
  globus_rls_client_error_info(r, NULL, errmsg, MAXERRMSG, GLOBUS_FALSE);
  fprintf(stderr, "%s\n", errmsg);
}
showbulkfailures(rl);
cleanexit(0); 
}
#endif

  if (interactive) {
    while (!quitting) {
      if (isatty(fileno(stdin)))
	printf("rls> ");
      if (!fgets(buf, BUFSIZ, stdin))
	break;
      lex(buf, &iargc, iargv);
      r = pcmd(iargc, iargv);
      if (r != GLOBUS_SUCCESS) {
	globus_rls_client_error_info(r, NULL, errmsg, MAXERRMSG, GLOBUS_FALSE);
	fprintf(stderr, "%s\n", errmsg);
      }
    }
    cleanexit(0);
  }

  r = pcmd(argc - optind - 1, &argv[optind]);
  if (r != GLOBUS_SUCCESS) {
    globus_rls_client_error_info(r, NULL, errmsg, MAXERRMSG, GLOBUS_FALSE);
    fprintf(stderr, "%s\n", errmsg);
    cleanexit(1);
  }

  cleanexit(0);
}

static void
showexists(char *objtype, char *val)

{
  char *existfmt  = "+ %s\n";
  printf(existfmt, val);
}

static void
showexistsbulk(globus_list_t *l, SHOW show)

{
  globus_list_t			*p;
  globus_rls_string2_bulk_t	*str2bulk;

  char *existfmt  = "+ %s\n";
  char *nexistfmt = "- %s\n";

  for (p = l; p; p = globus_list_rest(p)) {
    str2bulk = (globus_rls_string2_bulk_t *) globus_list_first(p);
    if (show == rls_cli_show_all) {
      char *fmt = (str2bulk->rc == GLOBUS_RLS_SUCCESS) ? existfmt : nexistfmt;
      printf(fmt, str2bulk->str2.s1);
    }
    else if (show == rls_cli_show_exists && str2bulk->rc == GLOBUS_RLS_SUCCESS)
      printf(existfmt, str2bulk->str2.s1);
    else if (show == rls_cli_show_nexists && str2bulk->rc != GLOBUS_RLS_SUCCESS)
      printf(nexistfmt, str2bulk->str2.s1);
  } /*endfor*/
  
  globus_rls_client_free_list(l);
}

static void
showresult(globus_list_t *l)

{
  globus_list_t			*p;
  globus_rls_string2_t	*str2;

  char *fmt = (decorate) ? "  %s: %s\n" : "  %s %s\n";

  for (p = l; p; p = globus_list_rest(p)) {
    str2 = (globus_rls_string2_t *) globus_list_first(p);
    printf(fmt, str2->s1, str2->s2);
  }
  globus_rls_client_free_list(l);
}

static void
showresultbulk(globus_list_t *l)

{
  globus_list_t				*p;
  globus_rls_string2_bulk_t	*str2bulk;
  char						errmsg[MAXERRMSG];

  char *fmt = (decorate) ? "  %s: %s\n" : "  %s %s\n";

  for (p = l; p; p = globus_list_rest(p)) {
    str2bulk = (globus_rls_string2_bulk_t *) globus_list_first(p);
    if (str2bulk->rc == GLOBUS_RLS_SUCCESS)
      printf(fmt, str2bulk->str2.s1, str2bulk->str2.s2);
    else
      printf(fmt, str2bulk->str2.s1,
	     globus_rls_errmsg(str2bulk->rc, NULL, errmsg, MAXERRMSG));
  }
  globus_rls_client_free_list(l);
}

static void
showbulkfailures(globus_list_t *l)

{
  globus_list_t				*p;
  globus_rls_string2_bulk_t	*str2bulk;
  char						errmsg[MAXERRMSG];

  char *fmt = (decorate) ? "  %s,%s: %s\n" : "  %s %s %s\n";

  for (p = l; p; p = globus_list_rest(p)) {
    str2bulk = (globus_rls_string2_bulk_t *) globus_list_first(p);
    printf(fmt, str2bulk->str2.s1, str2bulk->str2.s2,
	   globus_rls_errmsg(str2bulk->rc, NULL, errmsg, MAXERRMSG));
  }
  globus_rls_client_free_list(l);
}

static void
showattr(globus_list_t *l)

{
  globus_list_t				*p;
  globus_rls_attribute_t	*a;
  char						buf[BUFSIZ];

  char *fmt = (decorate) ? "  %s: %s: %s\n" : "  %s %s %s\n";

  for (p = l; p; p = globus_list_rest(p)) {
    a = (globus_rls_attribute_t *) globus_list_first(p);
    printf(fmt, a->name, attr2a(a->type),
	   globus_rls_client_attr2s(a, buf, BUFSIZ));
  }
  globus_rls_client_free_list(l);
}

static void
showattrdef(globus_list_t *l)

{
  globus_list_t				*p;
  globus_rls_attribute_t	*a;

  char *fmt = (decorate) ? "  %s: %s\n" : "  %s %s\n";

  for (p = l; p; p = globus_list_rest(p)) {
    a = (globus_rls_attribute_t *) globus_list_first(p);
    printf(fmt, a->name, attr2a(a->type));
  }
  globus_rls_client_free_list(l);
}

static void
showattrobj(globus_list_t *l)

{
  globus_list_t					*p;
  globus_rls_attribute_object_t	*ao;
  char							buf[BUFSIZ];

  char *fmt = (decorate) ? "  %s: %s: %s: %s\n" : "  %s %s %s %s\n";
  char *errfmt = (decorate) ? "  %s: %s: %s\n" : "  %s %s %s\n";

  for (p = l; p; p = globus_list_rest(p)) {
    ao = (globus_rls_attribute_object_t *) globus_list_first(p);
    if (ao->rc == GLOBUS_RLS_SUCCESS)
      printf(fmt, ao->key, ao->attr.name, attr2a(ao->attr.type),
	     globus_rls_client_attr2s(&ao->attr, buf, BUFSIZ));
    else
      printf(errfmt, ao->key, ao->attr.name,
	     globus_rls_errmsg(ao->rc, NULL, buf, BUFSIZ));
  }
  globus_rls_client_free_list(l);
}

static int
a2obj(char *s, globus_rls_obj_type_t *ot)

{
  if (strcasecmp(s, "lfn") == 0)
    *ot = globus_rls_obj_lrc_lfn;
  else if (strcasecmp(s, "pfn") == 0)
    *ot = globus_rls_obj_lrc_pfn;
  else {
    fprintf(stderr, "Unknown object type %s: should be lfn or pfn\n", s);
    return 0;
  }
  return 1;
}

static int
a2attr(char *s, globus_rls_attr_type_t *at)

{
  int	i;

  if ((i = findkwd(s, attrtypekwds)) == -1)
    return 0;
  *at = attrtypekwds[i].narg;	/* Kludge! */
  return 1;
}

static int
a2show(char *s, SHOW *show)

{
  int  i;

  if ((i = findkwd(s, showexistskwds)) == -1)
    return 0;
  *show = showexistskwds[i].narg;	/* Kludge! */
  return 1;
}

static int
a2op(char *s, globus_rls_attr_op_t *op)

{
  if (strcmp(s, "=") == 0)
    *op = globus_rls_attr_op_eq;
  else if (strcmp(s, "!=") == 0)
    *op = globus_rls_attr_op_ne;
  else if (strcmp(s, ">") == 0)
    *op = globus_rls_attr_op_gt;
  else if (strcmp(s, ">=") == 0)
    *op = globus_rls_attr_op_ge;
  else if (strcmp(s, "<") == 0)
    *op = globus_rls_attr_op_lt;
  else if (strcmp(s, "<=") == 0)
    *op = globus_rls_attr_op_le;
  else if (strcasecmp(s, "like") == 0)
    *op = globus_rls_attr_op_like;
  else {
    fprintf(stderr, "Unknown operator %s: should be =, !=, >, >=, <, <=, like\n", s);
    return 0;
  }
  return 1;
}

static char *
attr2a(globus_rls_attr_type_t type)

{
  switch (type) {
    case globus_rls_attr_type_date:	return "date";
    case globus_rls_attr_type_flt:	return "float";
    case globus_rls_attr_type_int:	return "int";
    case globus_rls_attr_type_str:	return "string";
  }
  return "unknown";
}

static globus_result_t
pcmd(int argc, char **argv)

{
  int	ai;
  int	i;
  int	j;
  KWD	*k;

  k = cmdkwds;
  ai = 0;
  cmdi = 0;
  while (ai < argc) {
    i = findkwd(argv[ai], k);
    if (i == -1)
      return GLOBUS_SUCCESS;	/* Error already dealt with	*/
    cmds[cmdi++] = k[i].kwd;
    ai++;
    if (k[i].nextkwd) {
      k = k[i].nextkwd;
      continue;
    }
    if (argc - ai != k[i].narg && k[i].narg >= 0) {
      if (argc - ai < k[i].narg)
	fprintf(stderr, "Incomplete command:");
      else
	fprintf(stderr, "Too many arguments:");
      for (j = 0; j < cmdi; j++)
	fprintf(stderr, " %s", cmds[j]);
      if (k[i].arghelp)
	fprintf(stderr, " %s", k[i].arghelp);
      fprintf(stderr, "\n");
      return GLOBUS_SUCCESS;
    }
    return k[i].f(argc - ai, &argv[ai]);
  }
  if (cmdi) {
    fprintf(stderr, "Incomplete command:\n");
    showcmds(cmdi, cmds, k);
  }
  return GLOBUS_SUCCESS;
}

static globus_result_t
doadd(int argc, char **argv)

{
  return globus_rls_client_lrc_add(h, argv[0], argv[1]);
}

static globus_result_t
doattradd(int argc, char **argv)

{
  globus_result_t		r;
  globus_rls_attribute_t	a;

  if (!a2obj(argv[2], &a.objtype) || !a2attr(argv[3], &a.type))
    return GLOBUS_SUCCESS;
  if ((r = globus_rls_client_s2attr(a.type, argv[4], &a)) != GLOBUS_SUCCESS)
    return r;
  a.name = argv[1];
  return globus_rls_client_lrc_attr_add(h, argv[0], &a);
}

static globus_result_t
doattrdelete(int argc, char **argv)

{
  globus_rls_attribute_t	a;

  if (!a2obj(argv[2], &a.objtype))
    return GLOBUS_SUCCESS;
  a.name = argv[1];
  return globus_rls_client_lrc_attr_remove(h, argv[0], &a);
}


/* Bulk Data Input */
#define MAX_BULK_ELEMENTS        (500)
#define MAX_BULK_ARGS            6
#define MAX_BULK_LINE_LENGTH     (1*1024)
#define BULK_INIT(ac, av, step)  init_bulk_input(ac, av, step)
#define BULK_NEXT(x)             ((x)->next(x))
#define BULK_FREE_RECENT(x)      ((x)->infile?(x)->pos=0:0)
#define BULK_FREE(x)             free_bulk_input(x)
typedef struct bulk_input {
  /* file input */
  FILE *infile;
  char *inbuf;
  char *fargv[MAX_BULK_ARGS+1];
  int  lineno;

  /* argv input */
  char **argv;
  int  step;
  int  max;

  /* both */
  int  pos;
  char **(*next)(struct bulk_input *in);
} bulk_input;

static char **
bulk_next_args_from_argv(bulk_input *in)

{
  char **rv = NULL;
  int pos = in->pos;

  if (pos < in->max) {
    rv = &(in->argv[pos]);
    pos += in->step;
    in->pos = pos;
    return rv;
  }
  return NULL;
}

static char **
bulk_next_args_from_file(bulk_input *in)

{
  char *line;
  char *ptr;
  int i;

  assert(in->pos < MAX_BULK_ELEMENTS);
  line=(in->inbuf)+((MAX_BULK_LINE_LENGTH+1) * in->pos);
  in->pos++;

  if (feof(in->infile)) {
    return NULL;
  }
  ptr = fgets(line, MAX_BULK_LINE_LENGTH, in->infile);
  if (!ptr && feof(in->infile)) {
    return NULL;
  }
  if (ptr && !feof(in->infile) && ptr[strlen(ptr)-1]!='\n') { /* long line check */
    fprintf(stderr, "ERROR: Input line length exceeds %d on line %d\n", 
            MAX_BULK_LINE_LENGTH, in->lineno);
    cleanexit(1);
  }
  in->fargv[0]=strtok(ptr, " \t\r\n");
  for (i=1; i < in->step; i++) {
    if (! (in->fargv[i]=strtok(NULL, " \t\r\n")) ) {
      fprintf(stderr, "ERROR: Input line too short. Not enough args on line %d\n", in->lineno);
      cleanexit(1);
    }
  }
  in->lineno++;
  return in->fargv;
}

static void
init_bulk_argv_input(int argc, char **argv, int step, bulk_input *s)

{
  s->lineno = 0;
  s->max = argc;
  s->step = step;
  s->pos = 0;
  s->argv = argv;
  s->next = bulk_next_args_from_argv;
  s->infile = NULL;
}

static void
init_bulk_file_input(FILE *f, int step, bulk_input *s)

{
  s->lineno = 1;
  s->infile = f;
  s->inbuf = (char *)malloc( MAX_BULK_ELEMENTS * (MAX_BULK_LINE_LENGTH+1) );
  s->pos = 0;
  s->step = step;
  s->next = bulk_next_args_from_file;
  s->fargv[0] = NULL;
}

static bulk_input *
init_bulk_input(int argc, char **argv, int step)

{
  bulk_input *in = (bulk_input *) malloc(sizeof(bulk_input));
  if (infile) {
    init_bulk_file_input(infile, step, in);
    if (argc)
      fprintf(stderr, "WARNING: ignoring command line values in favor of file iput.\n");
  }
  else {
    init_bulk_argv_input(argc, argv, step, in);
  }
  return in;
}

static void
free_bulk_input(bulk_input *in)

{
  if (in->infile)
    free(in->inbuf);
  free(in);
}

/* END Bulk Data Input */


static globus_result_t
doattrbulkadd(int argc, char **argv)

{
  globus_result_t		r;
  globus_list_t			*al = NULL;
  globus_list_t			*rl;
  globus_rls_attribute_object_t	*ao;
  globus_rls_obj_type_t		ot;
  globus_rls_attr_type_t	at;

  int           n;
  bulk_input    *in;

  in = BULK_INIT(argc, argv, 5);
  n = 0;
  do {
    argv = BULK_NEXT(in);
    if (argv) {
      if (!a2obj(argv[2], &ot) || !a2attr(argv[3], &at)) {
        globus_list_free(al);
        return GLOBUS_SUCCESS;
      }
      ao = (globus_rls_attribute_object_t *) malloc(sizeof(globus_rls_attribute_object_t));
      ao->key = argv[0];
      ao->attr.name = argv[1];
      ao->attr.objtype = ot;
      ao->attr.type = at;
      if ((r = globus_rls_client_s2attr(ao->attr.type, argv[4], &ao->attr)) != GLOBUS_SUCCESS) {
        globus_list_free(al);
        free(ao);
        break;
      }
      globus_list_insert(&al, ao);
      n++;
    }
    if ( (n >= MAX_BULK_ELEMENTS) || (!argv && n) ) {
      n = 0;
      if ((r = globus_rls_client_lrc_attr_add_bulk(h, al, &rl)) == GLOBUS_SUCCESS)
        showbulkfailures(rl);
      freelistdatums(al);
      al = NULL;
      rl = NULL;
      BULK_FREE_RECENT(in);
      if (r != GLOBUS_SUCCESS)
        break;
    }
  } while (argv);
  BULK_FREE(in);
  return r;
}

static globus_result_t
doattrbulkdelete(int argc, char **argv)

{
  globus_result_t		r;
  globus_list_t			*al = NULL;
  globus_list_t			*rl;
  globus_rls_attribute_object_t	*ao;
  globus_rls_obj_type_t		ot;
  globus_rls_attr_type_t	at;

  int           n;
  bulk_input    *in;

  in = BULK_INIT(argc, argv, 5);
  n = 0;
  do {
    argv = BULK_NEXT(in);
    if (argv) {
      if (!a2obj(argv[2], &ot) || !a2attr(argv[3], &at)) {
        globus_list_free(al);
        return GLOBUS_SUCCESS;
      }
      ao = (globus_rls_attribute_object_t *) malloc(sizeof(globus_rls_attribute_object_t));
      ao->key = argv[0];
      ao->attr.name = argv[1];
      ao->attr.objtype = ot;
      ao->attr.type = at;
      if ((r = globus_rls_client_s2attr(ao->attr.type, argv[4], &ao->attr)) != GLOBUS_SUCCESS) {
        globus_list_free(al);
        free(ao);
        break;
      }
      globus_list_insert(&al, ao);
      n++;
    }
    if ( (n >= MAX_BULK_ELEMENTS) || (!argv && n) ) {
      n = 0;
      if ((r = globus_rls_client_lrc_attr_remove_bulk(h, al, &rl)) == GLOBUS_SUCCESS)
        showbulkfailures(rl);
      freelistdatums(al);
      al = NULL;
      rl = NULL;
      BULK_FREE_RECENT(in);
      if (r != GLOBUS_SUCCESS)
        break;
    }
  } while (argv);
  BULK_FREE(in);
  return r;
}

static globus_result_t
doattrbulkquery(int argc, char **argv)

{
  globus_result_t		r;
  globus_list_t			*al = NULL;
  globus_list_t			*rl;
  globus_rls_obj_type_t		ot;
  char				*attr;

  int           n;
  bulk_input    *in;

  if (!a2obj(argv[1], &ot))
    return GLOBUS_SUCCESS;
  if (strcmp(argv[0], "-") == 0)
    attr = NULL;
  else
    attr = argv[0];

  in = BULK_INIT(argc-2, argv+2, 1);
  n = 0;
  do {
    argv = BULK_NEXT(in);
    if (argv) {
      n++;
      globus_list_insert(&al, argv[0]);
    }
    if ( (n >= MAX_BULK_ELEMENTS) || (!argv && n) ) {
      n = 0;
      if ((r = globus_rls_client_lrc_attr_value_get_bulk(h, al, attr, ot, &rl)) == GLOBUS_SUCCESS)
        showattrobj(rl);
      globus_list_free(al);
      rl = NULL;
      al = NULL;
      BULK_FREE_RECENT(in);
      if (r != GLOBUS_SUCCESS)
        break;
    }
  } while (argv);
  BULK_FREE(in);
  return r;
}

static globus_result_t
doattrmodify(int argc, char **argv)

{
  globus_rls_attribute_t	a;
  globus_result_t		r;

  if (!a2obj(argv[2], &a.objtype) || !a2attr(argv[3], &a.type))
    return GLOBUS_SUCCESS;
  a.name= argv[1];
  if ((r = globus_rls_client_s2attr(a.type, argv[4],
				    &a)) != GLOBUS_SUCCESS)
    return r;
  return globus_rls_client_lrc_attr_modify(h, argv[0], &a);
}

static globus_result_t
doattrsearch(int argc, char **argv)

{
  globus_rls_obj_type_t		ot;
  globus_rls_attr_op_t		op;
  globus_rls_attribute_t	op1;
  globus_result_t		r;
  globus_list_t			*l;
  int				offset;

  if (!a2obj(argv[1],&ot) || !a2op(argv[2],&op) || !a2attr(argv[3],&op1.type))
    return GLOBUS_SUCCESS;
  if ((r = globus_rls_client_s2attr(op1.type, argv[4],
				    &op1)) != GLOBUS_RLS_SUCCESS)
    return r;
  offset = 0;
  while ((r = globus_rls_client_lrc_attr_search(h, argv[0], ot, op, &op1,
		NULL, &offset, reslimit, &l)) == GLOBUS_SUCCESS) {
    showattrobj(l);
    if (offset == -1)
      break;
  }
  return r;
}

static globus_result_t
dobulkadd(int argc, char **argv)

{
  globus_result_t		r;
  globus_rls_string2_t		*str2;
  globus_list_t			*al = NULL;
  globus_list_t			*rl;

  bulk_input    *in;
  int           n;

  in = BULK_INIT(argc, argv, 2);
  n = 0;
  do {
    argv = BULK_NEXT(in);
    if (argv) {
      n++;
      str2 = (globus_rls_string2_t *) malloc(sizeof(globus_rls_string2_t));
      str2->s1 = argv[0];
      str2->s2 = argv[1];
      globus_list_insert(&al, str2);
    }
    if ( (n >= MAX_BULK_ELEMENTS) || (!argv && n) ) {
      n = 0;
      if ((r = globus_rls_client_lrc_add_bulk(h, al, &rl)) == GLOBUS_SUCCESS)
        showbulkfailures(rl);
      freelistdatums(al);
      rl = NULL;
      al = NULL;
      BULK_FREE_RECENT(in);
      if (r != GLOBUS_SUCCESS)
        break;
    }
  } while (argv);
  BULK_FREE(in);
  return r;
}

static globus_result_t
dobulkcreate(int argc, char **argv)

{
  globus_result_t		r;
  globus_rls_string2_t		*str2;
  globus_list_t			*al = NULL;
  globus_list_t			*rl;

  int        n=0;
  bulk_input *in;

  in = BULK_INIT(argc, argv, 2);
  n = 0;
  do {
    argv = BULK_NEXT(in);
    if (argv) {
      n++;
      str2 = (globus_rls_string2_t *) malloc(sizeof(globus_rls_string2_t));
      str2->s1 = argv[0];
      str2->s2 = argv[1];
      globus_list_insert(&al, str2);
    }
    if ( (n>=MAX_BULK_ELEMENTS) || (!argv && n) ) {
      n=0;
      if ((r = globus_rls_client_lrc_create_bulk(h, al, &rl)) == GLOBUS_SUCCESS)
        showbulkfailures(rl);  /* Frees rl as a side effect */
      freelistdatums(al);
      rl = NULL;
      al = NULL;
      BULK_FREE_RECENT(in);
      if (r != GLOBUS_SUCCESS)
        break;
    }
  } while (argv);
  BULK_FREE(in);
  return r;
}

static globus_result_t
dobulkdelete(int argc, char **argv)

{
  globus_result_t		r;
  globus_rls_string2_t		*str2;
  globus_list_t			*al = NULL;
  globus_list_t			*rl;

  int           n;
  bulk_input    *in;
  in = BULK_INIT(argc, argv, 2);
  n=0;
  do {
    argv = BULK_NEXT(in);
    if (argv) {
      n++;
      str2 = (globus_rls_string2_t *) malloc(sizeof(globus_rls_string2_t));
      str2->s1 = argv[0];
      str2->s2 = argv[1];
      globus_list_insert(&al, str2);
    }
    if ( (n>=MAX_BULK_ELEMENTS) || (!argv && n) ) {
      n=0;
      if ((r = globus_rls_client_lrc_delete_bulk(h, al, &rl)) == GLOBUS_SUCCESS)
        showbulkfailures(rl);
      freelistdatums(al);
      rl = NULL;
      al = NULL;
      if (r != GLOBUS_SUCCESS)
        break;
      BULK_FREE_RECENT(in);
    }
  } while (argv);
  BULK_FREE(in);
  return r;
}

static globus_result_t
dobulkexistslrclfn(int argc, char **argv)

{
  globus_result_t	r;
  globus_list_t		*al = NULL;
  globus_list_t		*l;
  SHOW			show;

  int n;
  bulk_input *in;

  if (!a2show(argv[0], &show))
    return GLOBUS_SUCCESS;

  in = BULK_INIT(argc-1, argv+1, 1);
  n = 0;
  do {
    argv = BULK_NEXT(in);
    if (argv) {
      n++;
      globus_list_insert(&al, argv[0]);
    }
    if ( (n>=MAX_BULK_ELEMENTS) || (!argv && n) ) {
      n = 0;
      if ((r = globus_rls_client_lrc_exists_bulk(h, al, globus_rls_obj_lrc_lfn, &l)) == GLOBUS_SUCCESS)
        showexistsbulk(l, show);
      globus_list_free(al);
      l = NULL;
      al = NULL;
      BULK_FREE_RECENT(in);
      if (r != GLOBUS_SUCCESS)
        break;
    }
  } while (argv);
  BULK_FREE(in);
  return r;
}

static globus_result_t
dobulkexistslrcpfn(int argc, char **argv)

{
  globus_result_t	r;
  globus_list_t		*al = NULL;
  globus_list_t		*l;
  SHOW			show;

  int n;
  bulk_input *in;

  if (!a2show(argv[0], &show))
    return GLOBUS_SUCCESS;

  in = BULK_INIT(argc-1, argv+1, 1);
  n = 0;
  do {
    argv = BULK_NEXT(in);
    if (argv) {
      n++;
      globus_list_insert(&al, argv[0]);
    }
    if ( (n>=MAX_BULK_ELEMENTS) || (!argv && n) ) {
      n = 0;
      if ((r = globus_rls_client_lrc_exists_bulk(h, al, globus_rls_obj_lrc_pfn, &l)) == GLOBUS_SUCCESS)
        showexistsbulk(l, show);
      globus_list_free(al);
      l = NULL;
      al = NULL;
      BULK_FREE_RECENT(in);
      if (r != GLOBUS_SUCCESS)
        break;
    }
  } while (argv);
  BULK_FREE(in);
  return r;
}

static globus_result_t
dobulkexistsrlilfn(int argc, char **argv)

{
  globus_result_t	r;
  globus_list_t		*al = NULL;
  globus_list_t		*l;
  SHOW			show;

  int n;
  bulk_input *in;

  if (!a2show(argv[0], &show))
    return GLOBUS_SUCCESS;

  in = BULK_INIT(argc-1, argv+1, 1);
  n = 0;
  do {
    argv = BULK_NEXT(in);
    if (argv) {
      n++;
      globus_list_insert(&al, argv[0]);
    }
    if ( (n>=MAX_BULK_ELEMENTS) || (!argv && n) ) {
      n = 0;
      if ((r = globus_rls_client_rli_exists_bulk(h, al, globus_rls_obj_rli_lfn, &l)) == GLOBUS_SUCCESS)
        showexistsbulk(l, show);
      globus_list_free(al);
      l = NULL;
      al = NULL;
      BULK_FREE_RECENT(in);
      if (r != GLOBUS_SUCCESS)
        break;
    }
  } while (argv);
  BULK_FREE(in);
  return r;
}

static globus_result_t
dobulkquerylrclfn(int argc, char **argv)

{
  globus_result_t	r;
  globus_list_t		*al = NULL;
  globus_list_t		*l;

  int           n;
  bulk_input    *in;

  in = BULK_INIT(argc, argv, 1);
  n = 0;
  do {
    argv = BULK_NEXT(in);
    if (argv) {
      n++;
      globus_list_insert(&al, argv[0]);
    }
    if ( (n>=MAX_BULK_ELEMENTS) || (!argv && n) ) {
      n = 0;
      if ((r = globus_rls_client_lrc_get_pfn_bulk(h, al, &l)) == GLOBUS_SUCCESS)
        showresultbulk(l);
      globus_list_free(al);
      l = NULL;
      al = NULL;
      BULK_FREE_RECENT(in);
      if (r != GLOBUS_SUCCESS)
        break;
    }
  } while (argv);
  BULK_FREE(in);
  return r;
}

static globus_result_t
dobulkquerylrcpfn(int argc, char **argv)

{
  globus_result_t	r;
  globus_list_t		*al = NULL;
  globus_list_t		*l;

  int           n;
  bulk_input    *in;

  in = BULK_INIT(argc, argv, 1);
  n = 0;
  do {
    argv = BULK_NEXT(in);
    if (argv) {
      n++;
      globus_list_insert(&al, argv[0]);
    }
    if ( (n>=MAX_BULK_ELEMENTS) || (!argv && n) ) {
      n = 0;
      if ((r = globus_rls_client_lrc_get_lfn_bulk(h, al, &l)) == GLOBUS_SUCCESS)
        showresultbulk(l);
      globus_list_free(al);
      l = NULL;
      al = NULL;
      BULK_FREE_RECENT(in);
      if (r != GLOBUS_SUCCESS)
        break;
    }
  } while (argv);
  BULK_FREE(in);
  return r;
}

static globus_result_t
dobulkqueryrlilfn(int argc, char **argv)

{
  globus_result_t	r;
  globus_list_t		*al = NULL;
  globus_list_t		*l;

  int           n;
  bulk_input    *in;

  in = BULK_INIT(argc, argv, 1);
  n = 0;
  do {
    argv = BULK_NEXT(in);
    if (argv) {
      n++;
      globus_list_insert(&al, argv[0]);
    }
    if ( (n>=MAX_BULK_ELEMENTS) || (!argv && n) ) {
      n = 0;
      if ((r = globus_rls_client_rli_get_lrc_bulk(h, al, &l)) == GLOBUS_SUCCESS)
        showresultbulk(l);
      globus_list_free(al);
      l = NULL;
      al = NULL;
      BULK_FREE_RECENT(in);
      if (r != GLOBUS_SUCCESS)
        break;
    }
  } while (argv);
  BULK_FREE(in);
  return r;
}

static globus_result_t
dobulkrenamelfn(int argc, char **argv)

{
  globus_result_t		r;
  globus_rls_string2_t		*str2;
  globus_list_t			*al = NULL;
  globus_list_t			*rl;

  bulk_input    *in;
  int           n;

  in = BULK_INIT(argc, argv, 2);
  n = 0;
  do {
    argv = BULK_NEXT(in);
    if (argv) {
      n++;
      str2 = (globus_rls_string2_t *) malloc(sizeof(globus_rls_string2_t));
      str2->s1 = argv[0];
      str2->s2 = argv[1];
      globus_list_insert(&al, str2);
    }
    if ( (n >= MAX_BULK_ELEMENTS) || (!argv && n) ) {
      n = 0;
      if ((r = globus_rls_client_lrc_renamelfn_bulk(h, al, &rl)) == GLOBUS_SUCCESS)
        showbulkfailures(rl);
      freelistdatums(al);
      rl = NULL;
      al = NULL;
      BULK_FREE_RECENT(in);
      if (r != GLOBUS_SUCCESS)
        break;
    }
  } while (argv);
  BULK_FREE(in);
  return r;
}

static globus_result_t
dobulkrenamepfn(int argc, char **argv)

{
  globus_result_t		r;
  globus_rls_string2_t		*str2;
  globus_list_t			*al = NULL;
  globus_list_t			*rl;

  bulk_input    *in;
  int           n;

  in = BULK_INIT(argc, argv, 2);
  n = 0;
  do {
    argv = BULK_NEXT(in);
    if (argv) {
      n++;
      str2 = (globus_rls_string2_t *) malloc(sizeof(globus_rls_string2_t));
      str2->s1 = argv[0];
      str2->s2 = argv[1];
      globus_list_insert(&al, str2);
    }
    if ( (n >= MAX_BULK_ELEMENTS) || (!argv && n) ) {
      n = 0;
      if ((r = globus_rls_client_lrc_renamepfn_bulk(h, al, &rl)) == GLOBUS_SUCCESS)
        showbulkfailures(rl);
      freelistdatums(al);
      rl = NULL;
      al = NULL;
      BULK_FREE_RECENT(in);
      if (r != GLOBUS_SUCCESS)
        break;
    }
  } while (argv);
  BULK_FREE(in);
  return r;
}

static globus_result_t
doclear(int argc, char **argv)

{
  return globus_rls_client_lrc_clear(h);
}

static globus_result_t
docreate(int argc, char **argv)

{
  return globus_rls_client_lrc_create(h, argv[0], argv[1]);
}

static globus_result_t
dodelete(int argc, char **argv)

{
  return globus_rls_client_lrc_delete(h, argv[0], argv[1]);
}

static globus_result_t
dodefine(int argc, char **argv)

{
  globus_rls_obj_type_t		ot;
  globus_rls_attr_type_t	at;

  if (!a2obj(argv[1], &ot) || !a2attr(argv[2], &at))
    return GLOBUS_SUCCESS;
  return globus_rls_client_lrc_attr_create(h, argv[0], ot, at);
}

static globus_result_t
doexit(int argc, char **argv)

{
  quitting++;
  return GLOBUS_SUCCESS;
}

static globus_result_t
dohelp(int argc, char **argv)

{
  showcmds(0, NULL, cmdkwds);
  return GLOBUS_SUCCESS;
}

static globus_result_t
doqueryattr(int argc, char **argv)

{
  globus_rls_obj_type_t	ot;
  globus_result_t	r;
  globus_list_t		*l;
  char			*attr;

  if (!a2obj(argv[2], &ot))
    return GLOBUS_SUCCESS;
  if (strcmp(argv[1], "-") == 0)
    attr = NULL;
  else
    attr = argv[1];
  if ((r = globus_rls_client_lrc_attr_value_get(h, argv[0], attr, ot,
						&l)) == GLOBUS_SUCCESS)
    showattr(l);
  return r;
}

static globus_result_t
doquerylrclfn(int argc, char **argv)

{
  globus_result_t	r;
  globus_list_t		*l;
  int			offset;

  offset = 0;
  while ((r = globus_rls_client_lrc_get_pfn(h, argv[0], &offset, reslimit,
					    &l)) == GLOBUS_SUCCESS) {
    showresult(l);
    if (offset == -1)
      break;
  }
  return r;
}

static globus_result_t
doquerylrcpfn(int argc, char **argv)

{
  globus_result_t	r;
  globus_list_t		*l;
  int			offset;

  offset = 0;
  while ((r = globus_rls_client_lrc_get_lfn(h, argv[0], &offset, reslimit,
					    &l)) == GLOBUS_SUCCESS) {
    showresult(l);
    if (offset == -1)
      break;
  }
  return r;
}

static globus_result_t
doqueryrlilfn(int argc, char **argv)

{
  globus_result_t	r;
  globus_list_t		*l;
  int			offset;

  offset = 0;
  while ((r = globus_rls_client_rli_get_lrc(h, argv[0], &offset, reslimit,
					    &l)) == GLOBUS_SUCCESS) {
    showresult(l);
    if (offset == -1)
      break;
  }
  return r;
}

static globus_result_t
doquerywclrclfn(int argc, char **argv)

{
  globus_result_t	r;
  globus_list_t		*l;
  int			offset;

  offset = 0;
  while ((r = globus_rls_client_lrc_get_pfn_wc(h, argv[0], pattype,
		      &offset, reslimit, &l)) == GLOBUS_SUCCESS) {
    showresult(l);
    if (offset == -1)
      break;
  }
  return r;
}

static globus_result_t
doquerywclrcpfn(int argc, char **argv)

{
  globus_result_t	r;
  globus_list_t		*l;
  int			offset;

  offset = 0;
  while ((r = globus_rls_client_lrc_get_lfn_wc(h, argv[0], pattype,
			&offset, reslimit, &l)) == GLOBUS_SUCCESS) {
    showresult(l);
    if (offset == -1)
      break;
  }
  return r;
}

static globus_result_t
doquerywcrlilfn(int argc, char **argv)

{
  globus_result_t	r;
  globus_list_t		*l;
  int			offset;

  offset = 0;
  while ((r = globus_rls_client_rli_get_lrc_wc(h, argv[0], pattype,
			&offset, reslimit, &l)) == GLOBUS_SUCCESS) {
    showresult(l);
    if (offset == -1)
      break;
  }
  return r;
}

static globus_result_t
doshowattr(int argc, char **argv)

{
  globus_rls_obj_type_t	ot;
  globus_result_t	r;
  globus_list_t		*l;
  char			*attr;

  if (!a2obj(argv[1], &ot))
    return GLOBUS_SUCCESS;
  if (strcmp(argv[0], "-") == 0)
    attr = NULL;
  else
    attr = argv[0];
  if ((r = globus_rls_client_lrc_attr_get(h, attr, ot,
					  &l)) == GLOBUS_SUCCESS)
    showattrdef(l);
  return r;
}

static globus_result_t
dosetclearvalues(int argc, char **argv)

{
  clearvalues = strcasecmp(argv[0], "true") == 0;
  return GLOBUS_SUCCESS;
}

static globus_result_t
dosetreslimit(int argc, char **argv)

{
  reslimit = atoi(argv[0]);
  return GLOBUS_SUCCESS;
}

static globus_result_t
dosettimeout(int argc, char **argv)

{
  timeout = atoi(argv[0]);
  globus_rls_client_set_timeout(timeout);
  return GLOBUS_SUCCESS;
}

static globus_result_t
doundefine(int argc, char **argv)

{
  globus_rls_obj_type_t		ot;

  if (!a2obj(argv[1], &ot))
    return GLOBUS_SUCCESS;
  return globus_rls_client_lrc_attr_delete(h, argv[0], ot, clearvalues);
}

static globus_result_t
doexistslrc(int argc, char **argv)

{
  globus_rls_obj_type_t         ot;
  globus_result_t       	r;
  
  if (!a2obj(argv[0], &ot))
    return GLOBUS_SUCCESS;
  
  if ((r = globus_rls_client_lrc_exists(h, argv[1], ot)) == GLOBUS_SUCCESS)
    showexists(argv[0], argv[1]);

  return r;
}

static globus_result_t
doexistsrlilfn(int argc, char **argv)

{
  globus_result_t       	r;

  if ((r = globus_rls_client_rli_exists(h, argv[0], globus_rls_obj_rli_lfn)) == GLOBUS_SUCCESS)
    showexists("lfn", argv[0]);

  return r;
}

static globus_result_t
dorenamelfn(int argc, char **argv)

{
  return globus_rls_client_lrc_renamelfn(h, argv[0], argv[1]);
}

static globus_result_t
dorenamepfn(int argc, char **argv)

{
  return globus_rls_client_lrc_renamepfn(h, argv[0], argv[1]);
}

static void
lex(char *buf, int *argc, char **argv)

{
  char	*cp;
  int	quoted;

  *argc = 0;
  cp = buf;
  while (1) {
    while (isspace(*cp))
      cp++;
    if (!*cp)
      return;
    if (*cp == '"') {
      quoted = 1;
      cp++;
    } else
      quoted = 0;
    argv[(*argc)++] = cp;
    if (quoted)
      while (*cp && *cp != '"')
	cp++;
    else
      while (!isspace(*cp))
        cp++;
    *cp++ = '\0';
  }
}

static int
findkwd(char *kwd, KWD *kwdtab)

{
  int	len;
  int	i;
  int	r;

  len = strlen(kwd);
  for (i = 0; kwdtab[i].kwd; i++) {
    r = strncasecmp(kwd, kwdtab[i].kwd, len);
    if (r < 0)
      break;
    if (r == 0) {
      if (!kwdtab[i+1].kwd || strlen(kwdtab[i].kwd) == len)
	return i;
      if (strncasecmp(kwd, kwdtab[i+1].kwd, len) == 0) {
	fprintf(stderr, "Ambiguous keyword '%s', matches %s and %s\n",
		kwd, kwdtab[i].kwd, kwdtab[i+1].kwd);
	return -1;
      }
      return i;
    }
  }
  fprintf(stderr, "Unknown keyword '%s', expected one of:", kwd);
  for (i = 0; kwdtab[i].kwd; i++)
    fprintf(stderr, " %s", kwdtab[i].kwd);
  fprintf(stderr, "\n");

  return -1;
}

static void
hsig(int s)

{
  quitting++;
}

static void
cleanexit(int s)

{
  int	i;

  if (h)
    globus_rls_client_close(h);
  for (i = NMODS - 1; i >= 0; i--)
    globus_module_deactivate(modules[i]);
  exit(s);
}

static void
showcmds(int argc, char **argv, KWD *k)

{
  int	i;
  int	j;
  char	*targv[MAXCMD];

  if (!argv)
    argv = targv;
  for (i = 0; k[i].kwd; i++) {
    if (k[i].nextkwd) {
      argv[argc] = k[i].kwd;
      showcmds(argc+1, argv, k[i].nextkwd);
    } else {
      fprintf(stderr, "  ");
      for (j = 0; j < argc; j++)
	fprintf(stderr, "%s ", argv[j]);
      fprintf(stderr, "%s ", k[i].kwd);
      if (k[i].arghelp)
	fprintf(stderr, "%s", k[i].arghelp);
      fprintf(stderr, "\n");
    }
  }
}

static void
freelistdatums(globus_list_t *l)

{
  globus_list_t	*p;

  for (p = l; p; p = globus_list_rest(p))
    free(globus_list_first(p));
  globus_list_free(l);
}

static void
usage(char *prog)

{
  fprintf(stderr, "Usage: %s [-c] [-h] [-i input-file] [-l result-limit] [-s] [-t timeout] [-u] [-v] [command] rls-url\n", prog);
  showcmds(0, NULL, cmdkwds);
  cleanexit(1);
}
