/*
** 1998-12-19 -	Let's pretend we're one of those big, bad, desktop environments, and provide
**		users with an "Information" window in which we can show cool stuff... This
**		will basically repeat what stat() has told us, plus a few extras.
** 1999-03-06 -	Changed for the new selection/dirrow handling.
** 1999-03-28 -	Now uses the progress-indicating version of the fut_dir_size() routine.
** 1999-04-05 -	Added use of the new command configuration system, to allow user to have
**		some control over how this command operates. Nice.
** 2000-07-02 -	Internationalized. Lost some hard-coded plural suffix stuff (was English-only). :(
** 2002-02-08 -	Now uses the dialog module. Why it wasn't, nobody knows.
*/

#include "gentoo.h"

#include <time.h>

#include "errors.h"
#include "file.h"
#include "dialog.h"
#include "dirpane.h"
#include "iconutil.h"
#include "sizeutil.h"
#include "strutil.h"
#include "fileutil.h"
#include "userinfo.h"
#include "progress.h"
#include "cmdseq_config.h"

#include "cmd_info.h"

#define	CMD_ID	"info"

/* ----------------------------------------------------------------------------------------- */

typedef struct {
	gboolean modified;
	gboolean use_file;			/* Show 'file' output? */
	gboolean recurse_dirs;			/* Recurse directories? */
	gchar	 df_access[DP_DATEFMT_SIZE];	/* Formatter for dates of last access, */
	gchar	 df_modify[DP_DATEFMT_SIZE];	/*  modification, */
	gchar	 df_change[DP_DATEFMT_SIZE];	/*  and change. */
} OptInfo;

static OptInfo	info_options;
static CmdCfg	*info_cmc = NULL;

/* ----------------------------------------------------------------------------------------- */

static void build_date(const gchar *label, gint y, time_t *date, const gchar *fmt, GtkWidget *tab)
{
	gchar		buf[64];
	GtkWidget	*lab;
	struct tm	*tm;

	lab = gtk_label_new(label);
	gtk_table_attach(GTK_TABLE(tab), lab, 0, 1, y, y + 1, 0,0,5,0);

	if((tm = localtime(date)) != NULL)
	{
		if(strftime(buf, sizeof buf, fmt, tm) >= 0)
		{
			lab = gtk_label_new(buf);
			gtk_table_attach(GTK_TABLE(tab), lab, 1, 2, y, y + 1, 0,0,5,0);
		}
	}
}

static GtkWidget * build_info(MainInfo *min, DirPane *dp, DirRow *row)
{
	gchar		buf[128], sbuf[32], *ptr;
	guint		y;
	const gchar	*iname;
	GtkWidget	*tab, *ico, *lab, *szlab, *sep;
	GdkPixmap	*ipix;
	GdkBitmap	*imsk;

	tab = gtk_table_new(17, 2, FALSE);

	y = 0;
	if(DP_ROW_TYPE(row)->style != NULL)
	{
		if((iname = stl_style_property_get_icon(DP_ROW_TYPE(row)->style, SPN_ICON_UNSEL)) != NULL)
		{
			ipix = ico_icon_get(min, iname, &imsk);
			ico = gtk_pixmap_new(ipix, imsk);
			gtk_table_attach(GTK_TABLE(tab), ico, 0, 1, y, y + 1, 0,0,5,5);
		}
	}
	lab = gtk_label_new(DP_ROW_NAME(row));
	gtk_table_attach(GTK_TABLE(tab), lab, 1, 2, y, y + 1, GTK_EXPAND,0,5,5);
	y++;

	/* For symbolic links, display name of the target, at least. */
	if(S_ISLNK(DP_ROW_LSTAT(row).st_mode))
	{
		GtkWidget	*scwin, *text;

		lab = gtk_label_new(_("Link To"));
		gtk_table_attach(GTK_TABLE(tab), lab, 0, 1, y, y + 1, 0,0,5,0);
		scwin = gtk_scrolled_window_new(NULL, NULL);
		gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scwin), GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);
		text = gtk_text_new(NULL, NULL);
		gtk_text_set_editable(GTK_TEXT(text), FALSE);
		gtk_text_insert(GTK_TEXT(text), NULL, NULL, NULL, DP_ROW_LINKNAME(row), strlen(DP_ROW_LINKNAME(row)));
		gtk_container_add(GTK_CONTAINER(scwin), text);
		gtk_table_attach(GTK_TABLE(tab), scwin, 1, 2, y, y + 1,  GTK_EXPAND|GTK_FILL,GTK_EXPAND|GTK_FILL,5,0);
		y++;
	}

	sep = gtk_hseparator_new();
	gtk_table_attach(GTK_TABLE(tab), sep, 0, 2, y, y + 1, GTK_EXPAND | GTK_FILL,0,0,5);
	y++;

	lab = gtk_label_new(_("Type"));
	gtk_table_attach(GTK_TABLE(tab), lab, 0, 1, y, y + 1, 0,0,5,0);
	lab = gtk_label_new(DP_ROW_TYPE(row)->name);
	gtk_table_attach(GTK_TABLE(tab), lab, 1, 2, y, y + 1, 0,0,5,0);
	y++;

	lab = gtk_label_new(_("Location"));
	gtk_table_attach(GTK_TABLE(tab), lab, 0, 1, y, y + 1, 0,0,5,0);
	lab = gtk_label_new(dp->dir.path);
	gtk_table_attach(GTK_TABLE(tab), lab, 1, 2, y, y + 1, 0,0,5,0);
	y++;

	lab = gtk_label_new(_("Size"));
	gtk_table_attach(GTK_TABLE(tab), lab, 0, 1, y, y + 1, 0,0,5,0);
	sze_put_size(DP_ROW_LSTAT(row).st_size, sbuf, sizeof sbuf, SZE_AUTO);
	if(DP_ROW_LSTAT(row).st_size < 1024)
		stu_strncpy(buf, sbuf, sizeof buf);
	else
		g_snprintf(buf, sizeof buf, _("%s (%u bytes)"), sbuf, (gsize) DP_ROW_LSTAT(row).st_size);
	szlab = gtk_label_new(buf);
	gtk_table_attach(GTK_TABLE(tab), szlab, 1, 2, y, y + 1, 0,0,5,0);
	y++;

	/* Building info for a directory, and recursing enabled? */
	if(S_ISDIR(DP_ROW_LSTAT(row).st_mode) && info_options.recurse_dirs)
	{
		gchar	buf[64], sbuf[64];
		guint64	bytes = 0;
		FUCount	fc;

		/* Collect information recursively. */
		if(fut_dir_size_progress(min, dp_full_name(dp, DP_ROW_INDEX(dp, row)), &bytes, &fc))
		{
			/* Replace contents of previously built size-label. */
			sze_put_size64(bytes, sbuf, sizeof buf, SZE_AUTO);
			g_snprintf(buf, sizeof buf, _("%s (%Lu bytes)"), sbuf, bytes);
			gtk_label_set_text(GTK_LABEL(szlab), buf);

			/* And tell a little something about the contents. */
			lab = gtk_label_new(_("Contains"));
			gtk_table_attach(GTK_TABLE(tab), lab, 0, 1, y, y + 1, 0,0,5,0);
			g_snprintf(buf, sizeof buf, _("%d dirs, %d files, %d symlinks"),
					fc.num_dirs, fc.num_files, fc.num_links);
			lab = gtk_label_new(buf);
			gtk_table_attach(GTK_TABLE(tab), lab, 1, 2, y, y + 1, 0,0,5,0);
			y++;
		}
	}

	/* Is the "Show 'file' Output?" option enabled? Requires "file -n -f -" to work. */
#if defined HAS_GOOD_FILE
	if(info_options.use_file)
	{
		gchar	buf[PATH_MAX];

		sep = gtk_hseparator_new();
		gtk_table_attach(GTK_TABLE(tab), sep, 0, 2, y, y + 1, GTK_EXPAND | GTK_FILL,0,0,5);
		y++;
		lab = gtk_label_new(_("'File' Info"));
		gtk_table_attach(GTK_TABLE(tab), lab, 0, 1, y, y + 1, 0,0,5,0);
		g_snprintf(buf, sizeof buf, "%s/%s", dp->dir.path, DP_ROW_NAME(row));
		lab = gtk_label_new(fle_file(min, buf));
		gtk_table_attach(GTK_TABLE(tab), lab, 1, 2, y, y + 1, 0,0,5,0);
		y++;
	}
#endif
	sep = gtk_hseparator_new();
	gtk_table_attach(GTK_TABLE(tab), sep, 0, 2, y, y + 1, GTK_EXPAND | GTK_FILL,0,0,5);
	y++;

	lab = gtk_label_new(_("Owner"));
	gtk_table_attach(GTK_TABLE(tab), lab, 0, 1, y, y + 1, 0,0,5,0);
	if((ptr = (gchar *) usr_lookup_uname(DP_ROW_LSTAT(row).st_uid)) != NULL)
		g_snprintf(buf, sizeof buf, "%s (%d)", ptr, (int) DP_ROW_LSTAT(row).st_uid);
	else
		g_snprintf(buf, sizeof buf, "%d", (int) DP_ROW_LSTAT(row).st_uid);
	lab = gtk_label_new(buf);
	gtk_table_attach(GTK_TABLE(tab), lab, 1, 2, y, y + 1, 0,0,5,0);
	y++;

	lab = gtk_label_new(_("Group"));
	gtk_table_attach(GTK_TABLE(tab), lab, 0, 1, y, y + 1, 0,0,5,0);
	if((ptr = (gchar *) usr_lookup_gname(DP_ROW_LSTAT(row).st_gid)) != NULL)
		g_snprintf(buf, sizeof buf, "%s (%d)", ptr, (int) DP_ROW_LSTAT(row).st_gid);
	else
		g_snprintf(buf, sizeof buf, "%d", (int) DP_ROW_LSTAT(row).st_gid);
	lab = gtk_label_new(buf);
	gtk_table_attach(GTK_TABLE(tab), lab, 1, 2, y, y + 1, 0,0,5,0);
	y++;

	sep = gtk_hseparator_new();
	gtk_table_attach(GTK_TABLE(tab), sep, 0, 2, y, y + 1, GTK_EXPAND | GTK_FILL,0,0,5);
	y++;

	build_date(_("Accessed"), y++, &DP_ROW_LSTAT(row).st_atime, info_options.df_access, tab);
	build_date(_("Modified"), y++, &DP_ROW_LSTAT(row).st_ctime, info_options.df_modify, tab);
	build_date(_("Changed"),  y++, &DP_ROW_LSTAT(row).st_mtime, info_options.df_change, tab);

	if(errno)
	{
		gtk_widget_destroy(tab);
		return NULL;
	}
	return tab;
}

/* ----------------------------------------------------------------------------------------- */

/* 1999-03-06 -	Create window showing all about <row>, which sits in <dp>. */
static gint info(MainInfo *min, DirPane *dp, DirRow *row)
{
	GtkWidget	*fr;

	if((fr = build_info(min, dp, row)) != NULL)
	{
		gtk_widget_show_all(fr);
		dlg_dialog_async_new(fr, DP_ROW_NAME(row), _("OK"), NULL, NULL);
	}
	return (fr != NULL) ? TRUE : FALSE;
}

/* ----------------------------------------------------------------------------------------- */

/* 1998-12-19 -	Show information about files in <src>. Knows about double-click, and handles
**		it like everybody else.
*/
int cmd_information(MainInfo *min, DirPane *src, DirPane *dst, CmdArg *ca)
{
	GSList	*slist, *iter;

	if((slist = dp_get_selection(src)) == NULL)
		return 1;

	pgs_progress_begin(min, _("Getting Information..."), PFLG_BUSY_MODE);
	for(iter = slist; iter != NULL; iter = g_slist_next(iter))
	{
		if(!info(min, src, DP_SEL_ROW(iter)))
			break;
		dp_unselect(src, DP_SEL_INDEX(src, iter));
	}
	pgs_progress_end(min);

	if(errno)
		err_set(min, errno, CMD_ID, iter ? DP_SEL_NAME(iter) : NULL);
	else
		err_clear(min);
	err_show(min);

	dp_free_selection(slist);

	return 1;
}

/* ----------------------------------------------------------------------------------------- */

/* 1999-04-05 -	Initialize configuration stuff. */
void cfg_information(MainInfo *min)
{
	if(info_cmc == NULL)
	{
		/* Initialize default option values. */
		info_options.modified	  = FALSE;
		info_options.use_file	  = FALSE;
		info_options.recurse_dirs = TRUE;
		stu_strncpy(info_options.df_access, "%Y-%m-%d %H:%M.%S", sizeof info_options.df_access);
		stu_strncpy(info_options.df_modify, "%Y-%m-%d %H:%M.%S", sizeof info_options.df_modify);
		stu_strncpy(info_options.df_change, "%Y-%m-%d %H:%M.%S", sizeof info_options.df_change);

		info_cmc = cmc_config_new("Information", &info_options);
		cmc_field_add_boolean(info_cmc, "modified", NULL, offsetof(OptInfo, modified));
#if defined HAS_GOOD_FILE
		cmc_field_add_boolean(info_cmc, "use_file", _("Show 'file' Output?"), offsetof(OptInfo, use_file));
#endif
		cmc_field_add_boolean(info_cmc, "recurse_dirs", _("Recurse Directories?"), offsetof(OptInfo, recurse_dirs));
		cmc_field_add_string(info_cmc, "df_access", _("Access Date Format"), offsetof(OptInfo, df_access), sizeof info_options.df_access);
		cmc_field_add_string(info_cmc, "df_modify", _("Modify Date Format"), offsetof(OptInfo, df_modify), sizeof info_options.df_modify);
		cmc_field_add_string(info_cmc, "df_change", _("Change Date Format"), offsetof(OptInfo, df_change), sizeof info_options.df_change);

		cmc_config_register(info_cmc);
	}
}
