/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"
#include "misc_string.h"
#include "misc_gui.h"
#include "misc_general.h"
#include "embed.h"
#include "window.h"
#include "bookmarks.h"
#include "favicon.h"
#include "bookmarks_editor_callbacks.h"
#include "prefs.h"
#include "context.h"
#include "dialog.h"
#include "gfilepicker.h"
#include "autocompletion.h"
#include "bookmarks_io.h"
#include "glade.h"

#include <string.h>
#include <time.h>
#include <libgnome/gnome-config.h>
#include <libgnome/gnome-i18n.h>
#include <libgnomeui/gnome-popup-menu.h>
#include <libgnomeui/gnome-dialog-util.h>
#include <libgnomeui/gnome-preferences.h>
#include <gtk/gtktogglebutton.h>
#include <gtk/gtkselection.h>
#include <gtk/gtkcheckmenuitem.h>
#include <gtk/gtktext.h>
#include <gdk/gdkkeysyms.h>
#include <gtk/gtktoolbar.h>

/* internal helper functions */
static void fetch_icon_recursively (BookmarkItem *b,
				    BookmarksEditorControls *controls);
static void bookmarks_editor_apply (BookmarksEditorControls *controls);
static void bookmarks_editor_set_text (GtkWidget *control, gchar *text, 
				       gboolean active);
static void bookmarks_editor_set_name (BookmarksEditorControls *controls, 
				       BookmarkItem *bookmark, 
				       gboolean active);
static void bookmarks_editor_set_url (BookmarksEditorControls *controls, 
				      BookmarkItem *bookmark, gboolean active);
static void bookmarks_editor_set_nick (BookmarksEditorControls *controls, 
				       BookmarkItem *bookmark, 
				       gboolean active);
static void bookmarks_editor_set_create_toolbar (BookmarksEditorControls 
						 *controls, 
						 BookmarkItem *bookmark, 
						 gboolean active);
static void bookmarks_editor_set_create_context_menu (BookmarksEditorControls 
						      *controls, 
						      BookmarkItem *bookmark, 
						      gboolean active);
static void bookmarks_editor_set_notes (BookmarksEditorControls *controls, 
					BookmarkItem *bookmark, 
					gboolean active);
static void bookmarks_editor_set_pixmap (BookmarksEditorControls *controls, 
					 BookmarkItem *bookmark, 
					 gboolean active);
static void bookmarks_editor_set_times (BookmarksEditorControls *controls,
					BookmarkItem *bookmark);
static void bookmarks_selection_prepare (GList *l,
					 BookmarksEditorControls *controls);
static void bookmarks_editor_copy_or_cut (BookmarksEditorControls *controls, 
					  gboolean cut);
static gint bookmarks_row_compare (GList *a, GList *b);
void
bookmarks_editor_key_press_event_cb (GtkWidget *widget,GdkEventKey *event,
				     BookmarksEditorControls *controls);
static void
bookmarks_editor_sort_cb_generic (BookmarksEditorControls *controls, 
				  gboolean recursive, gboolean folders_first);
static gboolean bookmarks_find_search (BookmarksFindDialog *dialog,
				       gboolean prev, gboolean wrap);
void bookmarks_find_prev_clicked_cb (GtkWidget *widget,
				     BookmarksFindDialog *dialog);
void bookmarks_find_next_clicked_cb (GtkWidget *widget,
				     BookmarksFindDialog *dialog);
void bookmarks_find_cancel_clicked_cb (GtkWidget *widget,
				       BookmarksFindDialog *dialog);
void bookmarks_find_entry_changed_cb (GtkEditable *editable,
				      BookmarksFindDialog *dialog);
void bookmarks_find_criterium1_deactivate_cb (GtkMenuShell *menushell,
					      BookmarksFindDialog *dialog);
void bookmarks_find_criterium2_deactivate_cb (GtkMenuShell *menushell,
					      BookmarksFindDialog *dialog);
#if 0
static gboolean 
bookmarks_find_can_search_prevnext (BookmarksFindDialog *dialog,
				    gboolean backward);
static void
bookmarks_find_update_prevnext_sensitivity (BookmarksFindDialog *dialog);
#endif

extern gboolean block_search_position;
extern gboolean block_docking;
						
void
bookmarks_editor_ctree_tree_select_row_cb (GtkCTree *ctree, GList *node,
					   gint column,
					   BookmarksEditorControls *controls)
{	
	BookmarkItem *b = controls->selection = 
		gtk_ctree_node_get_row_data (ctree, GTK_CTREE_NODE (node));
	if (!b) return;
	controls->lock_edit_controls = TRUE;

	if (controls->set_default_folder_menuitem != NULL)
	{
		gtk_widget_set_sensitive
			(controls->set_default_folder_menuitem,
			 (b != default_bookmarks_root) 
			 && BOOKMARK_ITEM_IS_FOLDER (b));
	}

	if (controls->notes_text)
	{
		gtk_editable_delete_text (GTK_EDITABLE (controls->notes_text),
					  0, -1);
	}

	switch (b->type)
	{
	case BM_SITE:
		bookmarks_editor_set_name                 (controls, b, TRUE);
		bookmarks_editor_set_url                  (controls, b, TRUE);
		bookmarks_editor_set_nick                 (controls, b, TRUE);
		bookmarks_editor_set_create_toolbar       (controls, b, FALSE);
		bookmarks_editor_set_create_context_menu  (controls, b, TRUE);
		bookmarks_editor_set_notes                (controls, b, TRUE);
		bookmarks_editor_set_pixmap               (controls, b, TRUE);
		break;

	case BM_FOLDER:
	case BM_AUTOBOOKMARKS:
		bookmarks_editor_set_name                 (controls, b, TRUE);
		bookmarks_editor_set_url                  (controls, b, FALSE);
		bookmarks_editor_set_nick                 (controls, b, FALSE);
		bookmarks_editor_set_create_toolbar       (controls, b, TRUE);
		bookmarks_editor_set_create_context_menu  (controls, b, TRUE);
		bookmarks_editor_set_notes                (controls, b, TRUE);
		bookmarks_editor_set_pixmap               (controls, b, TRUE);
		break;

	case BM_SEPARATOR:
		bookmarks_editor_set_name                 (controls, b, FALSE);
		bookmarks_editor_set_url                  (controls, b, FALSE);
		bookmarks_editor_set_nick                 (controls, b, FALSE);
		bookmarks_editor_set_create_toolbar       (controls, b, FALSE);
		bookmarks_editor_set_create_context_menu  (controls, b, FALSE);
		bookmarks_editor_set_notes                (controls, b, FALSE);
		bookmarks_editor_set_pixmap               (controls, b, FALSE);
		break;
	}
	bookmarks_editor_set_times (controls, b);
	controls->lock_edit_controls = FALSE;
	
	/* update search stuff */
	if (!block_search_position && controls->find_dialog)
		controls->search_position = node;	
}

void
bookmarks_editor_ctree_tree_unselect_row_cb (GtkCTree *ctree, GList *node,
					     gint column,
					     BookmarksEditorControls *controls)
{	
	controls->lock_edit_controls = TRUE;
	controls->selection = NULL;

	bookmarks_editor_set_name                 (controls, NULL, FALSE);
	bookmarks_editor_set_url                  (controls, NULL, FALSE);
	bookmarks_editor_set_nick                 (controls, NULL, FALSE);
	bookmarks_editor_set_create_toolbar       (controls, NULL, FALSE);
	bookmarks_editor_set_create_context_menu  (controls, NULL, FALSE);
	bookmarks_editor_set_notes                (controls, NULL, FALSE);
	bookmarks_editor_set_pixmap               (controls, NULL, FALSE);
	bookmarks_editor_set_times                (controls, NULL);

	controls->lock_edit_controls = FALSE;
}

void
bookmarks_editor_edited_cb (GtkEditable *editable,
			    BookmarksEditorControls *controls)
{
	if (!controls->lock_edit_controls && controls->selection) {
		BookmarkItem *b = controls->selection;
		gchar *name = NULL;
		gchar *url = NULL;
		gchar *smarturl = NULL;
		gchar *nick = NULL;
		gchar *pixmap_file = NULL;
		gboolean create_toolbar = FALSE;
		gboolean create_context_menu = FALSE;
		gchar *notes = NULL;
		gchar *utf8_str;

		/* read and translate data from the widgets */
		if (controls->name_entry)
		{
			gchar *tmp1, *tmp2;
			tmp1 = gtk_editable_get_chars 
				(GTK_EDITABLE (controls->name_entry), 0, -1);
			tmp2 = misc_string_escape_uline_accel (tmp1);
			utf8_str = mozilla_locale_to_utf8 (tmp2);
			g_free (tmp2);
			g_free (tmp1);
			name = misc_string_strip_newline (utf8_str);
			g_free (utf8_str);
		}
		if (controls->url_entry)
			url = gtk_editable_get_chars 
				(GTK_EDITABLE (controls->url_entry), 0, -1);
		if (controls->smarturl_entry)
			smarturl = gtk_editable_get_chars
				(GTK_EDITABLE (controls->smarturl_entry),
				 0, -1);
		if (controls->nick_entry)
		{
			gchar *tmp1, *tmp2;
			tmp1 = gtk_editable_get_chars 
				(GTK_EDITABLE (controls->nick_entry), 0, -1);
			tmp2 = misc_string_escape_uline_accel (tmp1);
			utf8_str = mozilla_locale_to_utf8 (tmp2);
			g_free (tmp2);
			g_free (tmp1);
			nick = utf8_str;
		}
		if (controls->pixmap_file_entry)
			pixmap_file = gtk_editable_get_chars 
				(GTK_EDITABLE (controls->pixmap_file_entry), 
				 0, -1);
		if (controls->create_toolbar_toggle)
			create_toolbar = gtk_toggle_button_get_active 
				(GTK_TOGGLE_BUTTON (
					controls->create_toolbar_toggle));
		if (controls->create_context_menu_toggle)
			create_context_menu = gtk_toggle_button_get_active 
				(GTK_TOGGLE_BUTTON (
					controls->create_context_menu_toggle));
		if (controls->notes_text) 
		{
			gchar *tmp1, *tmp2;
			tmp1 = gtk_editable_get_chars 
				(GTK_EDITABLE (controls->notes_text), 0, -1);
			tmp2 = misc_string_escape_uline_accel (tmp1);
			utf8_str = mozilla_locale_to_utf8 (tmp2);
			g_free (tmp2);
			g_free (tmp1);
			notes = utf8_str;
		}

		/* modify the bookmark */
		
		bookmarks_toolbars_check_update (b);

		switch (b->type) {
		case BM_SITE:
			if (controls->name_entry) 
				bookmark_set_name (b, name);
			if (controls->url_entry)
				bookmark_set_url (b, url);
			if (controls->smarturl_entry) 
				bookmark_set_smarturl (b, smarturl);
			if (controls->nick_entry) 
				bookmark_set_nick (b, nick);
			if (controls->create_context_menu_toggle)
				bookmark_set_create_context_menu 
					(b, create_context_menu);
			if (controls->pixmap_file_entry) 
				bookmark_set_pixmap (b, pixmap_file);
			if (controls->notes_text) 
				bookmark_set_notes (b, notes);
			break;
		case BM_FOLDER:
		case BM_AUTOBOOKMARKS:
			if (controls->name_entry) 
				bookmark_set_name (b, name);
			if (controls->create_toolbar_toggle)
				bookmark_set_create_toolbar
					(b, create_toolbar);
			if (controls->create_context_menu_toggle)
				bookmark_set_create_context_menu 
					(b, create_context_menu);
			if (controls->pixmap_file_entry) 
				bookmark_set_pixmap (b, pixmap_file);
			if (controls->notes_text) 
				bookmark_set_notes (b, notes);
			break;
		default:
			break;
		}
		bookmark_set_time_modified (b, time (NULL));
		bookmarks_editor_set_times (controls, b);
		bookmarks_dirty = TRUE;

		bookmarks_toolbars_check_update (b);

		/* free stuff */
		g_free (name);
		g_free (url);
		g_free (smarturl);
		g_free (nick);
		g_free (pixmap_file);
		g_free (notes);
	}
}

static void
bookmarks_editor_apply (BookmarksEditorControls *controls)
{
	if (bookmarks_dirty)
	{
		bookmarks_updated ();
	}
}

void
bookmarks_editor_revert_clicked_cb (GtkWidget *widget,
				    BookmarksEditorControls *controls)
{

	/* load bookmarks from disk */
	bookmarks_load ();

	/* refresh everywhere else */
	bookmarks_editor_apply (controls);
}

void 
bookmarks_editor_apply_clicked_cb (GtkButton *button,
				   BookmarksEditorControls *controls)
{
	bookmarks_editor_apply (controls);
}

void 
bookmarks_editor_accept_clicked_cb (GtkButton *button,
				    BookmarksEditorControls *controls)
{
	bookmarks_editor_hide_dialog (controls);
}

gboolean
bookmarks_editor_delete_event_cb (GtkWidget *widget, GdkEvent *event,
				  BookmarksEditorControls *controls)
{
	bookmarks_editor_accept_clicked_cb (NULL, controls);
	return TRUE;
}

void
remove_confirmation_check_toggled_cb (GtkWidget *toggle, 
				      BookmarksEditorControls *controls) 
{
	gboolean toggled = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON 
							(toggle));
	if (toggled)
	{
		eel_gconf_set_integer (CONF_BOOKMARKS_REMOVE_CONFIRMATION_MIN,
				       -1);
	}
	else
	{
		eel_gconf_set_integer (CONF_BOOKMARKS_REMOVE_CONFIRMATION_MIN,
				       2);
	}
}

void 
remove_bookmarks_ok_clicked_cb (GtkButton *button,
				BookmarksEditorControls *controls)
{
	BookmarkItem *fake_root = controls->root_bookmark;
	GList *l = fake_root->list;
	fake_root->list = NULL;

	bookmarks_editor_freeze_all ();
	bookmarks_remove_several (l);
	bookmarks_editor_thaw_all ();

	g_list_free (l);
	bookmarks_editor_remove_bookmarks_window_destroy (controls);
}

void 
remove_bookmarks_cancel_clicked_cb (GtkButton *button,
				    BookmarksEditorControls *controls)
{
	bookmarks_editor_remove_bookmarks_window_destroy (controls);
}

gboolean
remove_bookmarks_delete_event_cb (GtkWidget *widget, GdkEvent *event,
				  BookmarksEditorControls *controls)
{
	bookmarks_editor_remove_bookmarks_window_destroy (controls);
	return TRUE;
}

void 
bookmarks_editor_remove_clicked_cb (GtkButton *button,
				    BookmarksEditorControls *controls)
{
	GList *selection_list = NULL;
	GtkCList *clist = GTK_CLIST(controls->ctree);
	BookmarkItem *next_to_select = NULL;

	selection_list = bookmarks_get_selections_sorted (clist);

	if (controls->selection && !g_list_next(clist->selection) &&
	    controls->selection->parent) { 
		/* there was only one bookmark selected */
		GList *next_to_select_pos = g_list_next (
			g_list_find (controls->selection->parent->list,
				     controls->selection));
		if (next_to_select_pos)
			next_to_select = next_to_select_pos->data;
	}
	
	bookmarks_selection_prepare (selection_list, controls);
	
	bookmarks_editor_remove_bookmarks_window (selection_list);

	/* Not sure if this is useful now
	if (next_to_select)
		gtk_ctree_select (ctree, bookmarks_get_tree_item 
				  (next_to_select, controls));
	*/

	g_list_free (selection_list);
}

void 
bookmarks_editor_move_up_clicked_cb (GtkButton *button,
				     BookmarksEditorControls *controls)
{
	GtkCList *clist;
	GList *selections_sorted, *selection;
	BookmarkItem *b;

	clist = GTK_CLIST(controls->ctree);
	
	selections_sorted = bookmarks_get_selections_sorted(clist);
	selection = selections_sorted;

	bookmarks_editor_freeze_all ();
	while (selection)
	{
		b = gtk_ctree_node_get_row_data
			(GTK_CTREE (controls->ctree),
			 GTK_CTREE_NODE (selection->data));
		bookmarks_toolbars_check_update (b);
		if (bookmarks_move_bookmark (controls, b, 0))
		{
			bookmarks_dirty = TRUE;
			break;
		}
		/* need to call bookmarks_toolbars_check_update twice because 
		   the parent of the bookmark may change */
		bookmarks_toolbars_check_update (b);
		bookmarks_dirty = TRUE;
		selection = selection->next;
	}
	bookmarks_editor_thaw_all ();
	
	g_list_free (selections_sorted);
}

void 
bookmarks_editor_move_down_clicked_cb (GtkButton *button,
				       BookmarksEditorControls *controls) 
{
	GtkCList *clist;
	GList *selections_sorted, *selection;
	BookmarkItem *b;

	clist = GTK_CLIST(controls->ctree);
	
	selections_sorted = bookmarks_get_selections_sorted (clist);
	selection = selections_sorted = g_list_reverse (selections_sorted);

	bookmarks_editor_freeze_all ();
	while (selection)
	{
		b = gtk_ctree_node_get_row_data
			(GTK_CTREE(controls->ctree),
			 GTK_CTREE_NODE(selection->data));
		bookmarks_toolbars_check_update (b);
		if (bookmarks_move_bookmark(controls, b, 1))
		{
			bookmarks_dirty = TRUE;
			break;
		}
		/* need to call bookmarks_toolbars_check_update twice because 
		   the parent of the bookmark may change */
		bookmarks_toolbars_check_update (b);
		bookmarks_dirty = TRUE;
		selection = selection->next;
	}
	bookmarks_editor_thaw_all ();
	
	g_list_free (selections_sorted);
}

void 
bookmarks_editor_fetch_icon_clicked_cb (GtkButton *button,
					BookmarksEditorControls *controls) 
{
	GtkCList *clist;
	GList *selections_sorted, *selection;
	BookmarkItem *b;

	clist = GTK_CLIST (controls->ctree);
	
	selections_sorted = bookmarks_get_selections_sorted(clist);
	selection = selections_sorted;

	while (selection != NULL)
	{
		b = gtk_ctree_node_get_row_data
			(GTK_CTREE(controls->ctree),
			 GTK_CTREE_NODE(selection->data));
		fetch_icon_recursively (b, controls);
		selection = selection->next;
	}
	
	g_list_free (selections_sorted);
}

static
void fetch_icon_recursively (BookmarkItem *b, 
			     BookmarksEditorControls *controls)
{
	GList *l;

	switch (b->type)
	{
	case BM_FOLDER:
	case BM_AUTOBOOKMARKS:
		for (l = b->list; l != NULL; l = g_list_next (l))
		{
			if (!b->alias_of)
				fetch_icon_recursively 
					((BookmarkItem *)(l->data),
					 controls);
		}
		break;
		
	case BM_SITE:
		favicon_download (NULL, b, FAVICON_DOWNLOAD_HOST_ICON |
				           FAVICON_DOWNLOAD_OVERWRITE);
		break;

	case BM_SEPARATOR:
		break;
	}
}

void
bookmarks_editor_edit_toggled_cb (GtkWidget *toggle, 
				  BookmarksEditorControls *controls) 
{
	gboolean visible = TRUE;
	
	if (block_docking) return;
	
	g_return_if_fail (controls->edit_frame != NULL);
	
	if (GTK_IS_TOGGLE_BUTTON (toggle))
	{
		visible = gtk_toggle_button_get_active 
				(GTK_TOGGLE_BUTTON (toggle));
		if (controls->edit_menuitem != NULL)
		{
			gtk_check_menu_item_set_active 
				(GTK_CHECK_MENU_ITEM (controls->edit_menuitem),
				 visible);
		}
	}
	else if (GTK_IS_CHECK_MENU_ITEM (toggle))
	{
		visible = GTK_CHECK_MENU_ITEM (toggle)->active;
		if (controls->edit_button != NULL)
		{
			gtk_toggle_button_set_active 
				(GTK_TOGGLE_BUTTON (controls->edit_button),
				 visible);
		}
	}
	if (visible)
		gtk_widget_show (controls->edit_frame);
	else
		gtk_widget_hide (controls->edit_frame);
}

void
bookmarks_editor_add_alias_cb (GtkButton *button,
			       BookmarksEditorControls *controls) 
{
	BookmarkItem *orig = controls->selection;
	BookmarkItem *near = controls->selection;

	GtkCListDragPos insert_pos = GTK_CLIST_DRAG_INTO;

	if (near)
	{
		BookmarkItem *new = bookmarks_new_alias (orig);
		new->time_added = time (NULL);
		bookmarks_insert_bookmark (new, near, insert_pos);
		bookmarks_editor_place_tree_items (new);
		bookmarks_editor_select_bookmark (controls, new);
	}
}

void
bookmarks_editor_add_alias_root_cb (GtkButton *button,
			       BookmarksEditorControls *controls) 
{
	BookmarkItem *orig = controls->root_bookmark;
	BookmarkItem *new = bookmarks_new_alias (orig);
	BookmarkItem *near = controls->selection;
	GtkCListDragPos insert_pos = GTK_CLIST_DRAG_INTO;
	if (!near) near = controls->root_bookmark;
	new->time_added = time (NULL);
	bookmarks_insert_bookmark (new, near, insert_pos);
	bookmarks_editor_place_tree_items (new);
	bookmarks_editor_select_bookmark (controls, new);
}


void 
bookmarks_editor_new_item_clicked_cb (GtkButton *button,
				      BookmarksEditorControls *controls) 
{
	BookmarkItem *new = bookmarks_new_bookmark 
		(BM_SITE, TRUE, "",  "http://",
		 NULL, NULL, NULL);
	BookmarkItem *near = controls->selection;
	GtkCListDragPos insert_pos = GTK_CLIST_DRAG_INTO;

	new->time_added = time (NULL);

	if (!near) near = controls->root_bookmark;
	bookmarks_insert_bookmark (new, near, insert_pos);
	bookmarks_editor_place_tree_items (new);
	bookmarks_editor_select_bookmark (controls, new);

	/* Don't grab the focus if we're in the temp bookmarks window and the
	   edit frame isn't visible */
	if (controls->edit_frame)
	{
		if (!GTK_WIDGET_VISIBLE (controls->edit_frame))
			return;
	}
	if (controls->name_entry) 
	{
		gtk_widget_grab_focus (controls->name_entry);
	}
}

void 
bookmarks_editor_new_folder_clicked_cb (GtkButton *button,
					  BookmarksEditorControls *controls) 
{
	BookmarkItem *new = bookmarks_new_bookmark 
		(BM_FOLDER, TRUE, "", NULL, NULL, NULL, NULL);
	BookmarkItem *near = controls->selection;
	GtkCListDragPos insert_pos = GTK_CLIST_DRAG_INTO;

	new->time_added = time (NULL);

	if (!near) near = controls->root_bookmark;
	bookmarks_insert_bookmark (new, near, insert_pos);
	bookmarks_editor_place_tree_items (new);
	bookmarks_editor_select_bookmark (controls, new);

	/* Don't grab the focus if we're in the temp bookmarks window and the
	   edit frame isn't visible */
	if (controls->edit_frame)
	{
		if (!GTK_WIDGET_VISIBLE (controls->edit_frame))
			return;
	}
	if (controls->name_entry)
	{
		gtk_widget_grab_focus (controls->name_entry);
	}
}

void 
bookmarks_editor_new_autobm_clicked_cb (GtkButton *button,
					BookmarksEditorControls *controls) 
{
	BookmarkItem *new = bookmarks_new_bookmark 
		(BM_AUTOBOOKMARKS, TRUE, _("AutoBookmarks"), NULL,
		 NULL, NULL, NULL);
	BookmarkItem *near = controls->selection;
	GtkCListDragPos insert_pos = GTK_CLIST_DRAG_INTO;

	new->time_added = time (NULL);

	if (!near) near = controls->root_bookmark;
	bookmarks_insert_bookmark (new, near, insert_pos);
	bookmarks_editor_place_tree_items (new);
	bookmarks_editor_select_bookmark (controls, new);

	/* Don't grab the focus if we're in the temp bookmarks window and the
	   edit frame isn't visible */
	if (controls->edit_frame)
	{
		if (!GTK_WIDGET_VISIBLE (controls->edit_frame))
			return;
	}
	if (controls->name_entry)
	{
		gtk_widget_grab_focus (controls->name_entry);
	}
	autobookmarks_generate ();
}

void 
bookmarks_editor_new_separator_clicked_cb (GtkButton *button,
					   BookmarksEditorControls *controls) 
{
	gchar* utfstr = mozilla_locale_to_utf8(_("Separator"));
	BookmarkItem *new = bookmarks_new_bookmark 
		(BM_SEPARATOR, TRUE, utfstr, NULL, NULL, NULL, NULL);
	BookmarkItem *near = controls->selection;
	GtkCListDragPos insert_pos = GTK_CLIST_DRAG_INTO;
	g_free(utfstr);

	new->time_added = time (NULL);

	if (!near) near = controls->root_bookmark;
	bookmarks_insert_bookmark (new, near, insert_pos);
	bookmarks_editor_place_tree_items (new);
	bookmarks_editor_select_bookmark (controls, new);
}

static void
bookmarks_editor_handle_link_clicked (BookmarksEditorControls *controls, 
				      LinkState state)
{
	GaleonEmbed *embed = NULL, *newembed = NULL;
	if (!controls->selection || (controls->selection->type != BM_SITE))
		return;

	/* Check if the window that is selected is already (still) open */
	if (controls->window && !g_list_find (all_windows, controls->window))
		controls->window = NULL;
	if (controls->window)
		embed = controls->window->active_embed;

	bookmarks_set_visited (controls->selection);
	if (controls->selection->smarturl &&
	    strstr (controls->selection->smarturl, "%s"))
	{
		bookmarks_smart_bm_dialog (embed, &newembed,
					   controls->selection, state);
	}
	else
	{
		embed_activate_link (embed, &newembed,
				     controls->selection->url, state);
	}
	if ((controls->window) && (controls->window->active_embed == newembed))
		/* only grab the focus when the bookmark is opened in the 
		   active embed */
		embed_grab_focus (controls->window->active_embed);

	/* remember created window if necessary */
	if (!controls->window && newembed)
		controls->window = newembed->parent_window;
}

void
bookmarks_editor_go_clicked_cb (GtkButton *button, 
				BookmarksEditorControls *controls)
{
	bookmarks_editor_handle_link_clicked (controls, 0);
}

gboolean
bookmarks_editor_copy_button_press_cb (GtkWidget *widget, 
				       GdkEventButton *event,
				       BookmarksEditorControls *controls)
{
	if (event->button == 3) {
		bookmarks_editor_copy_clicked_cb
			(GTK_BUTTON (widget), controls);
	}
	return FALSE;
}

void
bookmarks_editor_copy_clicked_cb (GtkButton *button,
				  BookmarksEditorControls *controls)
{
	if (controls->selection) {
		GtkMenu *destination_menu;

		destination_menu = 
			bookmarks_create_copy_menu (controls->selection);
		gnome_popup_menu_do_popup_modal
			(GTK_WIDGET (destination_menu), 
			 misc_gui_menu_position_under_widget, 
			 GTK_WIDGET (button), NULL, NULL);
		gtk_object_unref (GTK_OBJECT (destination_menu));
		bookmarks_dirty = TRUE;
		bookmarks_tb_dirty = TRUE;
	}
}

void 
bookmarks_editor_ctree_tree_move_cb (GtkCTree *ctree, GtkCTreeNode *node,
				     GtkCTreeNode *new_parent_node, 
				     GtkCTreeNode *new_sibling_node, 
				     BookmarksEditorControls *controls)
{
	BookmarkItem *source = gtk_ctree_node_get_row_data (ctree, node);
	BookmarkItem *new_parent = gtk_ctree_node_get_row_data 
		(ctree, new_parent_node);
	BookmarkItem *new_sib = gtk_ctree_node_get_row_data
		(ctree, new_sibling_node);
	GList *new_sib_list_node;
	gint new_pos = 0;
	
	g_return_if_fail (source && source->parent);

	bookmarks_toolbars_check_update (source);

	if (!new_parent)
		new_parent = controls->root_bookmark;
		
	source->parent->list = g_list_remove (source->parent->list, source);
	bookmarks_update_alias (source->parent);
	source->parent = new_parent;
	
	new_sib_list_node = g_list_find (new_parent->list, new_sib);
	if (new_sib_list_node) {
		new_pos = g_list_position (new_parent->list, 
					   new_sib_list_node);
		new_parent->list = g_list_insert (new_parent->list, source,
						  new_pos);
	} else
		new_parent->list = g_list_append (new_parent->list, source);

	bookmarks_update_alias (source->parent);
	bookmarks_editor_place_tree_items (source);

	bookmarks_toolbars_check_update (source);

	/* DEBUG only */
#if 0
	g_assert (g_list_find (source->parent->list, source));
	g_assert (g_list_position 
		  (source->parent->list, g_list_find (source->parent->list, source)) 
		  == new_pos);
#endif
}

void
bookmarks_editor_bookmark_copy_cb (GtkMenuItem *menuitem, BookmarkItem *dest)
{
	BookmarkItem *item = gtk_object_get_user_data (GTK_OBJECT (menuitem));
	BookmarkItem *new;
	g_assert (item != NULL);
	g_assert (dest->type == BM_FOLDER);
	new = bookmarks_copy_bookmark (item);
	new->parent = dest;
	dest->list = g_list_append (dest->list, new);
	bookmarks_editor_place_tree_items (new);
}

gboolean 
bookmarks_editor_ctree_button_press_event_cb (GtkWidget *widget,
					      GdkEventButton  *event, 
					      BookmarksEditorControls *controls)
{
	gint row, column;
	gtk_clist_get_selection_info (GTK_CLIST (widget), event->x, event->y,
				      &row, &column);
	controls->last_pressed = gtk_ctree_node_get_row_data 
		(GTK_CTREE (widget), gtk_ctree_node_nth (
			GTK_CTREE (widget), row));

	if (controls->last_pressed == NULL)
		return FALSE;
	
	if (event->button == 3)
	{
		GaleonWindow *window = NULL;
		BookmarkItem *bi = controls->last_pressed;
		GtkWidget *popup = NULL;
		
		/* Check if the window that is selected is still open */
		if (controls->window && g_list_find (all_windows,
							controls->window))
			window = controls->window;
		else
			//FIXME: logic of using the last window is not ideal
			window = g_list_last (all_windows)->data;
		return_val_if_not_window (window, FALSE);

		if (BOOKMARK_ITEM_IS_FOLDER (bi))
		{
			popup = gtk_menu_new ();
			gtk_object_set_data (GTK_OBJECT(popup), "in_editor", 
					     GINT_TO_POINTER (TRUE));
			bookmarks_create_edit_items (GTK_MENU (popup),
						     controls);
			context_menu_add_seperator (GTK_MENU (popup));
			bookmarks_menu_create_base_extra_items 
							(GTK_MENU (popup), bi);
		}
		else if (bi->type == BM_SITE)
		{
			popup = gtk_menu_new ();
			gtk_object_set_data (GTK_OBJECT(popup), "in_editor", 
					     GINT_TO_POINTER (TRUE));
			bookmarks_create_edit_items (GTK_MENU (popup),
						     controls);
			context_menu_add_seperator (GTK_MENU (popup));
			bookmarks_item_create_base_extra_items 
							(GTK_MENU (popup), bi);
		}
		if (popup)
		{
			/* make sure that this entry is selected */
			GtkCTreeNode *binode = bookmarks_get_tree_item 
				(bi, controls);
			if (!g_list_find 
			    (GTK_CLIST (controls->ctree)->selection, binode))
			{
				gtk_clist_unselect_all 
					(GTK_CLIST (controls->ctree));
				gtk_ctree_select (GTK_CTREE (controls->ctree),
						  binode);
			}	

			/* warning: ugly code */
			gtk_object_set_data(GTK_OBJECT(popup), "GaleonWindow",
					    window);
			gnome_popup_menu_do_popup_modal (popup, NULL,
							 NULL, event, NULL);
			gtk_object_unref (GTK_OBJECT(popup));

			gtk_signal_emit_stop_by_name (GTK_OBJECT(widget),
					"button-press-event");
			return TRUE;
		}
	}
	/* double click or middle click opens the window */
	else if (event->button != 1 || event->type == GDK_2BUTTON_PRESS)
	{
		/* make sure that this entry is selected */
		gtk_ctree_select (GTK_CTREE (controls->ctree),
				  bookmarks_get_tree_item (
					  controls->last_pressed, controls));
		bookmarks_editor_handle_link_clicked (controls, 
				misc_general_mouse_state_to_link_state (
						event->button, event->state));
	}

	return FALSE;
}

gboolean
bookmarks_editor_ctree_key_press_event_cb (GtkWidget *widget,
					   GdkEventKey *event,
					   BookmarksEditorControls *controls)
{
	if (event->keyval == GDK_Return)
		bookmarks_editor_handle_link_clicked (controls,
			misc_general_key_state_to_link_state (event->state));
	else if (event->state == 0)
	{
		if (event->keyval == GDK_Delete ||
		    event->keyval == GDK_KP_Delete)
			bookmarks_editor_remove_clicked_cb(NULL, controls);
	}
	else if ( (event->state & GDK_LOCK_MASK ||
		   event->state & GDK_MOD2_MASK ||
		   event->state & GDK_MOD5_MASK) &&
		 !(event->state & GDK_SHIFT_MASK ||
		   event->state & GDK_MOD1_MASK  ||
		   event->state & GDK_CONTROL_MASK) )
	{
		if (event->keyval == GDK_Delete)
			bookmarks_editor_remove_clicked_cb(NULL, controls);
	}

	return FALSE;
}

void 
bookmarks_editor_tree_expand_cb (GtkCTree *ctree, GtkCTreeNode *node, 
				 BookmarksEditorControls *controls)
{
	BookmarkItem *b = gtk_ctree_node_get_row_data (ctree, node);
	if (b) {
		b->expanded = TRUE;
		bookmarks_dirty = TRUE;
	}
}

void 
bookmarks_editor_tree_collapse_cb (GtkCTree *ctree, GtkCTreeNode *node, 
				   BookmarksEditorControls *controls)
{
	BookmarkItem *b = gtk_ctree_node_get_row_data (ctree, node);
	if (b) {
		b->expanded = FALSE;
		bookmarks_dirty = TRUE;
	}
}

void 
bookmarks_editor_expand_all_cb (GtkButton *button,
				BookmarksEditorControls *controls) 
{
	GList *l;
	bookmarks_editor_freeze_all ();
	for (l = controls->root_bookmark->list; l != NULL; l = l->next) {
		BookmarkItem *b = l->data;
		gtk_ctree_expand_recursive 
			(GTK_CTREE (controls->ctree), 
			 bookmarks_get_tree_item (b, controls));
	}
	bookmarks_editor_thaw_all ();
}

void 
bookmarks_editor_collapse_all_clicked_cb (GtkButton *button,
					  BookmarksEditorControls *controls)
{
	GList *l;
	bookmarks_editor_freeze_all ();
	for (l = controls->root_bookmark->list; l != NULL; l = l->next) {
		BookmarkItem *b = l->data;
		gtk_ctree_collapse_recursive
			(GTK_CTREE (controls->ctree), 
			 bookmarks_get_tree_item (b, controls));
	}
	bookmarks_editor_thaw_all ();
}

/**
 * bookmarks_editor_set_text: generic bookmarks editor text field controller
 */
static void
bookmarks_editor_set_text (GtkWidget *control, gchar *text, gboolean active)
{
	/* skip if no control in this editor */
	if (control == NULL)
		return;

	/* set the text if active, or nullify */
	gtk_entry_set_text (GTK_ENTRY (control), active && text ? text : "");

	/* set widget sensitivity */
	gtk_widget_set_sensitive  (GTK_WIDGET (control), active);
}

/**
 * bookmarks_editor_set_name: set name field in bookmarks editor
 */
static void
bookmarks_editor_set_name (BookmarksEditorControls *controls, 
			   BookmarkItem *bookmark, gboolean active)
{
	gchar *tmp = NULL;
	gchar *loc_str = NULL;

	if (bookmark && bookmark->name) {
		tmp = misc_string_strip_uline_accel (bookmark->name);
		loc_str = mozilla_utf8_to_locale (tmp);
	}

	bookmarks_editor_set_text (controls->name_entry, 
				   bookmark ? loc_str : NULL, 
				   active);
	g_free (tmp);
	g_free (loc_str);
}

/**
 * bookmarks_editor_set_url: set url field in bookmarks editor
 */
static void
bookmarks_editor_set_url (BookmarksEditorControls *controls, 
			  BookmarkItem *bookmark, gboolean active)
{
	bookmarks_editor_set_text (controls->url_entry, 
				   bookmark ? bookmark->url : NULL, 
				   active);
	bookmarks_editor_set_text (controls->smarturl_entry,
				   bookmark ? bookmark->smarturl : NULL,
				   active);
}

/**
 * bookmarks_editor_set_nick: set nick field in bookmarks editor
 */
static void
bookmarks_editor_set_nick (BookmarksEditorControls *controls, 
			   BookmarkItem *bookmark, gboolean active)
{
	gchar *loc_str = NULL;

	if (!controls->nick_entry) return;

	if (bookmark)
		loc_str = mozilla_utf8_to_locale (bookmark->nick);

	bookmarks_editor_set_text (controls->nick_entry, 
				   loc_str, active);
	g_free (loc_str);
}

/**
 * bookmarks_editor_set_create_toolbar: set create toolbar toggle button
 * in bookmarks editor
 */
static void
bookmarks_editor_set_create_toolbar (BookmarksEditorControls *controls, 
				     BookmarkItem *bookmark, gboolean active)
{
	GtkWidget *control = controls->create_toolbar_toggle;

	/* check the control is present */
	if (control == NULL)
		return;

	/* set toggle button */
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (control),
				      active && bookmark &&
				      bookmark->create_toolbar);

	/* set widget sensitivity */
	gtk_widget_set_sensitive  (GTK_WIDGET (control), active);
}

/**
 * bookmarks_editor_set_create_context_menu: set create context menuu toggle
 * button in bookmarks editor
 */
static void
bookmarks_editor_set_create_context_menu (BookmarksEditorControls *controls, 
					  BookmarkItem *bookmark,
					  gboolean active)
{
	GtkWidget *control = controls->create_context_menu_toggle;

	/* check the control is present */
	if (control == NULL)
		return;

	/* set toggle button */
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (control),
				      active && bookmark &&
				      bookmark->create_context_menu);

	/* set widget sensitivity */
	gtk_widget_set_sensitive  (GTK_WIDGET (control), active);
}

/**
 * bookmarks_editor_set_notes: set notes field in bookmarks editor
 */
static void
bookmarks_editor_set_notes (BookmarksEditorControls *controls, 
			    BookmarkItem *bookmark, gboolean active)
{
	GtkWidget *control = controls->notes_text;

	/* check the control is present */
	if (control == NULL)
		return;

	/* set the text */
	gtk_editable_delete_text (GTK_EDITABLE (control), 0, -1);
	if (active && bookmark && bookmark->notes)
	{
		gchar *loc_str = mozilla_utf8_to_locale (bookmark->notes);
		gtk_text_insert (GTK_TEXT (control), NULL, NULL, NULL,
				 loc_str, strlen (bookmark->notes));
		g_free (loc_str);
	}

	/* set widget sensitivity */
	gtk_widget_set_sensitive  (GTK_WIDGET (control), active);
}

/**
 * bookmarks_editor_set_pixmap: set pixmap field in bookmarks editor
 */
static void
bookmarks_editor_set_pixmap (BookmarksEditorControls *controls, 
			     BookmarkItem *bookmark, gboolean active)
{
	bookmarks_editor_set_text (controls->pixmap_file_entry, 
				   bookmark ? bookmark->pixmap_file : NULL, 
				   active);
}

/* why does it need to look for the real bookmark? FIXME, VERIFY */
static void 
bookmarks_editor_set_times (BookmarksEditorControls *controls,
			    BookmarkItem *bookmark)
{
	gchar *text;
	BookmarkItem *bookmarkreal = NULL;
	if (bookmark)
		bookmarkreal = bookmarks_find_real_bookmark (bookmark);
	if (controls->time_added_entry) {
		text = bookmark ? misc_string_time_to_string (
						bookmark->time_added) 
			: g_strdup ("");
		gtk_entry_set_text (GTK_ENTRY (controls->time_added_entry),
				    text);
		g_free (text);
	}
	if (controls->time_modified_entry) {
		text = bookmarkreal ? misc_string_time_to_string
			(bookmarkreal->time_modified) : g_strdup ("");
		gtk_entry_set_text (GTK_ENTRY (controls->time_modified_entry),
				    text);
		g_free (text);
	}
	if (controls->time_visited_entry) {
		text = bookmarkreal 
			? misc_string_time_to_string (
						bookmarkreal->time_visited) 
			: g_strdup ("");
		gtk_entry_set_text (GTK_ENTRY (controls->time_visited_entry),
				    text);
		g_free (text);
	}
}

void 
bookmarks_editor_show_only_this_folder_cb (GtkButton *button,
				             BookmarksEditorControls *controls)
{
	if (controls->selection) {
		BookmarkItem *new_root = 
			BOOKMARK_ITEM_IS_FOLDER (controls->selection)
			? controls->selection
			: controls->selection->parent;
		if (new_root) bookmarks_editor_set_root (controls, new_root);
			
	}
}

void 
bookmarks_editor_show_all_bookmarks_cb (GtkButton *button,
				        BookmarksEditorControls *controls) 
{
	bookmarks_editor_set_root (controls, bookmarks_root);
}

void
bookmarks_editor_cut_cb (GtkButton *button,
			 BookmarksEditorControls *controls) 
{
	bookmarks_editor_copy_or_cut (controls, TRUE);
}

void
bookmarks_editor_copy_cb (GtkButton *button,
			  BookmarksEditorControls *controls) 
{
	bookmarks_editor_copy_or_cut (controls, FALSE);
}

static void
bookmarks_editor_copy_or_cut (BookmarksEditorControls *controls, 
			      gboolean cut)
{
	if (controls->selection) {
		BookmarkItem *dummy = bookmarks_new_bookmark 
			(BM_FOLDER, TRUE, "Dummy folder", 
			 NULL, NULL, NULL, NULL);
		GtkCList *clist = GTK_CLIST(controls->ctree);
		GList *selection_list = 
			bookmarks_get_selections_sorted (clist);
		bookmarks_selection_prepare (selection_list, controls);
		dummy->list = selection_list;
		if (controls->clipboard) g_free (controls->clipboard);
		controls->clipboard = bookmarks_item_to_string (dummy);
		gtk_selection_owner_set (GTK_WIDGET (clist),
					 GDK_SELECTION_PRIMARY,
					 GDK_CURRENT_TIME);
		dummy->list = NULL;
		bookmarks_free_bookmark (dummy);
		if (cut)
			bookmarks_remove_several (selection_list);
		g_list_free (selection_list);
	}
}

void
bookmarks_editor_paste_cb (GtkButton *button,
			   BookmarksEditorControls *controls) 
{
	gtk_selection_convert (GTK_WIDGET (controls->ctree), 
			       GDK_SELECTION_PRIMARY,
			       GDK_SELECTION_TYPE_STRING,
			       GDK_CURRENT_TIME);
}

void
bookmarks_editor_selection_get_cb (GtkWidget *widget, 
					GtkSelectionData *selection_data,
					guint info, guint time_stamp,
					BookmarksEditorControls *controls)
{
	if (controls->clipboard)
		gtk_selection_data_set (selection_data, 
					GDK_SELECTION_TYPE_STRING, // FIXME
					8, controls->clipboard, 
					strlen (controls->clipboard));
}

void 
bookmarks_selection_prepare (GList *l, BookmarksEditorControls *controls) 
{
	GtkCTree *ctree = GTK_CTREE (controls->ctree);
	GList *li;
	for (li = l; li; li = li->next) {
		BookmarkItem *b =  gtk_ctree_node_get_row_data 
			(ctree, li->data);
		li->data = b;
	}
}


void 
bookmarks_editor_selection_received_cb (GtkWidget *widget, 
					GtkSelectionData *selection_data,
					guint xtime)
{
	BookmarkItem *b;
	BookmarksEditorControls *controls = gtk_object_get_data 
		(GTK_OBJECT (widget), "controls");

	if (selection_data->length < 0)	{
		g_warning ("Selection retrieval failed\n");
		return;
	}

	b = bookmarks_item_from_string (selection_data->data);

	if (b) {
		BookmarkItem *near = controls->selection;
		GtkCListDragPos insert_pos = GTK_CLIST_DRAG_INTO;
		GList *li;
		if (!near) near = controls->root_bookmark;
		
		bookmarks_editor_freeze_all ();
		
		/* b is the dummy bookmark, we have to add all it's children */
		for (li = b->list; li; li = li->next)
		{
			BookmarkItem *new = li->data;
			new->time_modified = time (NULL);
			bookmarks_insert_bookmark (new, near, insert_pos);
			bookmarks_editor_place_tree_items (new);
			near = new;
			insert_pos = GTK_CLIST_DRAG_AFTER;
		}
		bookmarks_free_bookmark (b);
		bookmarks_dirty = TRUE;
		bookmarks_editor_thaw_all ();
	}
}

void
bookmarks_dock_close_clicked_cb (GtkWidget *button, 
				 BookmarksEditorControls *controls)
{	
	bookmarks_editor_apply (controls);
	bookmarks_editor_hide_dialog (controls);
}

void
bookmarks_dock_destroy_cb (GtkWidget *dialog, 
			   BookmarksEditorControls *controls)
{
	GaleonWindow *window;

	window = controls->window;
	return_if_not_window (window);

}

void 
bookmarks_dock_detach_button_clicked_cb (GtkButton *button,
					 BookmarksEditorControls *controls)
{
	GaleonWindow *window;

	window = controls->window;
	return_if_not_window (window);
	bookmarks_editor_apply (controls);

	bookmarks_editor_freeze_all ();
	bookmarks_editor_remove_tree_item (controls, 
			                   controls->root_bookmark);
	bookmarks_editor_thaw_all ();

	bookmarks_editors = g_list_remove (bookmarks_editors, controls);
	eel_gconf_set_boolean (CONF_STATE_BOOKMARKS_DOCK_ENABLE, FALSE);
	window_undock (window);
	bookmarks_editor_show_dialog (window);
}

void
bookmarks_editor_show_attach_cb (GtkWidget *toggle, 
				 BookmarksEditorControls *controls) 
{
	GaleonWindow *window;

	if (block_docking) return;

	window = controls->window;
	return_if_not_window (window);

	if (controls->type == STANDARD_BOOKMARKS_EDITOR)
	{
		eel_gconf_set_boolean (CONF_STATE_BOOKMARKS_DOCK_ENABLE, TRUE);
		bookmarks_editor_hide_dialog (controls);
		bookmarks_editor_show_dock (window);
	}
	else
	{
		eel_gconf_set_boolean (CONF_STATE_BOOKMARKS_DOCK_ENABLE, FALSE);
		bookmarks_editor_hide_dialog (controls);
		bookmarks_editor_show_dialog (window);
	}
}

void
bookmarks_editor_key_press_event_cb (GtkWidget *widget,GdkEventKey *event,
				     BookmarksEditorControls *controls)
{
	if (event->keyval == GDK_Escape)
	{
		bookmarks_editor_hide_dialog (controls);
	}
}

void
bookmarks_editor_set_default_folder_cb (GtkWidget *control, 
					BookmarksEditorControls *controls)
{
	g_return_if_fail (controls);

	if (controls->selection != NULL && 
	    BOOKMARK_ITEM_IS_FOLDER (controls->selection))
	{
		bookmarks_set_as_default_activate_cb 
			(NULL, controls->selection);
	}
}

void
bookmarks_editor_browse_clicked_cb (GnomeFileEntry *fileentry,
				    BookmarksEditorControls *controls)
{
	if (fileentry->fsw != NULL)
	{
		dialog_set_parent (fileentry->fsw, controls->dialog);
	}
}

static void
bookmarks_editor_toolbar_appearance_menu_cb (GtkRadioMenuItem *menuitem, 
					     BookmarksEditorControls *controls)
{
	GSList *list = g_slist_reverse
		(g_slist_copy (gtk_radio_menu_item_group (menuitem)));
	gint index = g_slist_index (list, menuitem);
	
	g_slist_free (list);

	gtk_toolbar_set_style (GTK_TOOLBAR (controls->toolbar), index);
	
	switch (controls->type)
	{
	case DOCKED_BOOKMARKS_EDITOR:
	        eel_gconf_set_integer 
			(CONF_TOOLBAR_BOOKMARKS_EDITOR_DOCKED_STYLE,
			 index);
		break;
	case STANDARD_BOOKMARKS_EDITOR:
		eel_gconf_set_integer 
			(CONF_TOOLBAR_BOOKMARKS_EDITOR_STYLE,
			 index);
		break;
	case REMOVE_BOOKMARKS_EDITOR:
		/* no toolbar in that case */
		break;
	}
	
	gtk_widget_queue_resize (controls->dialog);
}

gboolean
bookmarks_editor_toolbar_button_press_event_cb
                                          (GtkWidget *tb_event_box,
					   GdkEventButton *event,
					   BookmarksEditorControls *controls)
{
	GSList *radiogroup = NULL;
	gint style = 0;

	if (event->button == 3)
	{
		GtkMenu *popup = GTK_MENU (gtk_menu_new ());
	
		switch (controls->type)
		{
		case DOCKED_BOOKMARKS_EDITOR:
			style = eel_gconf_get_integer 
				 (CONF_TOOLBAR_BOOKMARKS_EDITOR_DOCKED_STYLE);
			break;
		case STANDARD_BOOKMARKS_EDITOR:
			style = eel_gconf_get_integer
				 (CONF_TOOLBAR_BOOKMARKS_EDITOR_STYLE);
			break;
		case REMOVE_BOOKMARKS_EDITOR:
			/* no toolbar in that case */
			break;
		}
	
		context_menu_add_radio_item 
			(popup, _("_Icons only"),
			 bookmarks_editor_toolbar_appearance_menu_cb, controls,
			 (style == GTK_TOOLBAR_ICONS), &radiogroup);
		context_menu_add_radio_item 
			(popup, _("_Text only"),
			 bookmarks_editor_toolbar_appearance_menu_cb, controls,
			 (style == GTK_TOOLBAR_TEXT), &radiogroup);
		context_menu_add_radio_item 
			(popup, _("Text _under icons"),
			 bookmarks_editor_toolbar_appearance_menu_cb, controls,
			 (style == GTK_TOOLBAR_BOTH), &radiogroup);
	
		gtk_signal_emit_stop_by_name (GTK_OBJECT(tb_event_box),
					      "button-press-event");

		gnome_popup_menu_do_popup_modal (GTK_WIDGET (popup), NULL,
						 NULL, event, NULL);
		gtk_object_unref (GTK_OBJECT (popup));
	}
	return TRUE;
}

void
bookmarks_editor_save_as_clicked_cb (GtkWidget *widget,
				     BookmarksEditorControls *controls)
{
	gchar *file = NULL;
	gint format_index;
	FileFormat *file_formats = bookmarks_io_get_fileformats ();

	if (show_file_picker (controls->dialog, _("Save as..."),
			      g_get_home_dir(), NULL, modeSave,
			      &file, NULL, file_formats, &format_index)) {
		gboolean ok;
		BookmarkTranslator *selected;
		selected = g_list_nth 
			(bookmarks_translators, format_index)->data;
		ok = selected->save_to_file 
			(bookmarks_root, default_bookmarks_root, file);
		if (!ok)
			gnome_error_dialog 
				(_("Couldn't save your bookmarks to the "
				   "specified file!"));
	}
	g_free (file);
	g_free (file_formats);
}

void
bookmarks_editor_open_clicked_cb (GtkWidget *widget, 
				       BookmarksEditorControls *controls)
{
	gchar *file = NULL;
	gint format_index;
	FileFormat *file_formats = bookmarks_io_get_fileformats ();

	if (show_file_picker (controls->dialog, _("Load from..."),
			      g_get_home_dir(), NULL, modeOpen,
			      &file, NULL, file_formats, &format_index)) 
	{
		gchar *question = g_strdup_printf
			(_("Your current set of bookmarks will be replaced "
			   "with the contents of %s.\n"
			   "Do you want to continue?"), file);
		if (dialog_ask_question (question, controls->dialog)) {
			BookmarkItem *new_root;
			BookmarkTranslator *selected;
			selected = g_list_nth 
				(bookmarks_translators, format_index)->data;
			default_bookmarks_root = NULL;
			new_root = selected->load_from_file 
				(file, &default_bookmarks_root);
			if (new_root)
				bookmarks_set_root (new_root);
			else 
			{
				gnome_error_dialog 
					(_("Couldn't load any bookmark from that file.\n"
					   "Make sure that it is in the correct file format."));
			}
		}
		g_free (question);
	}
	g_free (file);
	g_free (file_formats);

	bookmarks_dirty = TRUE;
	bookmarks_tb_dirty = TRUE;
	/* refresh everywhere else */
	bookmarks_editor_apply (controls);
}

void
bookmarks_editor_import_clicked_cb (GtkWidget *widget, 
				    BookmarksEditorControls *controls)
{
	gchar *file = NULL;
	gint format_index;
	FileFormat *file_formats = bookmarks_io_get_fileformats ();

	if (show_file_picker (controls->dialog, _("Import from..."),
			      g_get_home_dir(), NULL, modeOpen,
			      &file, NULL, file_formats, &format_index)) 
	{
		gchar *question = g_strdup_printf
			(_("Your current set of bookmarks will be merged "
			   "with the contents of %s.\n"
			   "Do you want to continue?"), file);
		if (dialog_ask_question (question, controls->dialog)) {
			BookmarkItem *new_root;
			BookmarkTranslator *selected;
			selected = g_list_nth 
				(bookmarks_translators, format_index)->data;
			new_root = selected->load_from_file (file, NULL);
			if (new_root)
				bookmarks_merge_trees (bookmarks_root, 
						       new_root);
			else
			{
				gnome_error_dialog 
					(_("Couldn't load any bookmark from that file.\n"
					   "Make sure that it is in the correct file format."));
			}
		}
		g_free (question);
	}
	g_free (file);
	g_free (file_formats);

	bookmarks_dirty = TRUE;
	bookmarks_tb_dirty = TRUE;
	/* refresh everywhere else */
	bookmarks_editors_refresh ();
	bookmarks_editor_apply (controls);
}

void
bookmarks_editor_sort_norec_clicked_cb (GtkWidget *widget, 
					BookmarksEditorControls *controls)
{
	bookmarks_editor_sort_cb_generic (controls, FALSE, FALSE);
}

void
bookmarks_editor_sort_rec_clicked_cb (GtkWidget *widget, 
				      BookmarksEditorControls *controls)
{
	bookmarks_editor_sort_cb_generic (controls, TRUE, FALSE);
}

void
bookmarks_editor_sort_ff_norec_clicked_cb (GtkWidget *widget, 
					BookmarksEditorControls *controls)
{
	bookmarks_editor_sort_cb_generic (controls, FALSE, TRUE);
}

void
bookmarks_editor_sort_ff_rec_clicked_cb (GtkWidget *widget, 
				      BookmarksEditorControls *controls)
{
	bookmarks_editor_sort_cb_generic (controls, TRUE, TRUE);
}

static void
bookmarks_editor_sort_cb_generic (BookmarksEditorControls *controls, 
				  gboolean recursive, gboolean folders_first)
{
	GList *l = recursive 
		? g_list_copy (GTK_CLIST (controls->ctree)->selection)
		: bookmarks_get_selections_sorted (GTK_CLIST(controls->ctree));
	GList *li;
	bookmarks_selection_prepare (l, controls);
	for (li = l; li != NULL; li = li->next) {
		BookmarkItem *sel = li->data;
		BookmarkItem *sort_root = BOOKMARK_ITEM_IS_FOLDER (sel)
			? sel 
			: sel->parent;
		if (sort_root) 
		{
			if (recursive) 
				bookmarks_sort_rec (sort_root, folders_first,
						    TRUE);
			else
				bookmarks_sort_norec (sort_root, folders_first,
						      TRUE);
		}
	}
	g_list_free (l);
}

/**
 * Sorts clist selections by row number and prunes any children in the
 * list whose ancestor is also in the list.  The returned GList should
 * be freed. 
 */
GList *
bookmarks_get_selections_sorted (GtkCList *clist)
{
	GtkCTree *ctree;
	GList *selections_sorted;
	GList *node, *child;
	gint length, i;
		
	ctree = GTK_CTREE(clist);
	selections_sorted = g_list_copy(clist->selection);

	if (!selections_sorted)
		return NULL;

	/* check for multiple selections. we want to preserve
	   the original order of the items, regardless of the
	   order in which they were selected, so the selection
	   list is sorted by row number */
	if (selections_sorted->next)
	{
		selections_sorted = g_list_sort(selections_sorted,
					(GCompareFunc) bookmarks_row_compare);
		length = g_list_length(selections_sorted);
		node = selections_sorted;

		/* prune the list to make sure we don't try to move
		   an item whose ancestor is also in the list */
		for (i = 0; i < length; i++)
		{
			gboolean remove_child = FALSE;
			GtkCTreeNode *del_node = NULL;
			
			child = selections_sorted;
			while (child)
			{
				if (gtk_ctree_is_ancestor(ctree,
					    GTK_CTREE_NODE(node->data),
					    GTK_CTREE_NODE(child->data)))
				{
					del_node = GTK_CTREE_NODE(child->data);
					remove_child = TRUE;
				}
				child = child->next;

				if (remove_child)
				{
					selections_sorted = g_list_remove(
							  selections_sorted,
							  del_node);
					remove_child = FALSE;
				}
			}
			if (!(node = g_list_nth(selections_sorted, i+1)))
				break;
		}
	}
	return (selections_sorted);
}

static gint bookmarks_row_compare (GList *a, GList *b)
{
	GList *root;
	gint row_a, row_b;
	
	root = g_list_first(a);
	row_a = g_list_position(root, a);
	row_b = g_list_position(root, b);
	
	if (row_a < row_b)
		return -1;
	else if (row_a == row_b)
		return 0;
	else
		return 1;
}

void
bookmarks_dock_menu_button_clicked_cb (GtkButton *button,
				       BookmarksEditorControls *controls)
{
	bookmarks_editor_popup_menu (controls, GTK_WIDGET (button));
}


void 
bookmarks_editor_destroy_cb (GtkWidget *widget, 
			     BookmarksEditorControls *controls)
{
	bookmarks_editor_hide_dialog (controls);
}

void
bookmarks_dock_search_entry_activate_cb (GtkWidget *widget,
					 BookmarksEditorControls *controls)
{
	gchar *text = gtk_editable_get_chars (GTK_EDITABLE (widget), 0, -1);
	gchar *utext = mozilla_locale_to_utf8(text);

	if (bookmarks_editor_search (controls, utext, BOOKMARKS_SEARCH_IN_NAME,
	    BOOKMARKS_SEARCH_METHOD_CONTAINS, FALSE, FALSE) == FALSE)
	{
		bookmarks_editor_search (controls, utext,
			 		 BOOKMARKS_SEARCH_IN_NAME,
					 BOOKMARKS_SEARCH_METHOD_CONTAINS,
					 FALSE, TRUE);
	}

	gtk_widget_grab_focus (widget);

	g_free (utext);
	g_free (text);
}

void
bookmarks_editor_find_bookmarks_cb (GtkWidget *widget,
				    BookmarksEditorControls *controls)
{
	BookmarksFindDialog *dialog;
	GladeXML *gxml;

	if (controls->find_dialog)
	{
		gdk_window_raise (controls->find_dialog->dialog->window);
		return;
	}
	
	/* build dialog */
	dialog = g_new0 (BookmarksFindDialog, 1);
	gxml = glade_widget_new ("bookmarks.glade", 
				 "bookmarks_find_dialog",
				 &(dialog->dialog), dialog);
	dialog->prev_button           = glade_xml_get_widget (gxml,
		"bookmarks_find_prev_button");
	dialog->next_button           = glade_xml_get_widget (gxml,
		"bookmarks_find_next_button");
	dialog->criterium1_optionmenu = glade_xml_get_widget (gxml,
		"bookmarks_find_criterium1_optionmenu");
	dialog->criterium2_optionmenu = glade_xml_get_widget (gxml,
		"bookmarks_find_criterium2_optionmenu");
	dialog->gnomeentry            = glade_xml_get_widget (gxml,
		"bookmarks_find_gnomeentry");
	dialog->entry                 = glade_xml_get_widget (gxml,
		"bookmarks_find_entry");

	g_assert (dialog->prev_button           != NULL);
	g_assert (dialog->next_button           != NULL);
	g_assert (dialog->criterium1_optionmenu != NULL);
	g_assert (dialog->criterium2_optionmenu != NULL);
	g_assert (dialog->gnomeentry            != NULL);
	g_assert (dialog->entry                 != NULL);

	/* connect signals */
	gtk_signal_connect (GTK_OBJECT
		(GTK_OPTION_MENU (dialog->criterium1_optionmenu)->menu),
		 "deactivate", bookmarks_find_criterium1_deactivate_cb,
		 dialog);
	gtk_signal_connect (GTK_OBJECT
		(GTK_OPTION_MENU (dialog->criterium2_optionmenu)->menu),
		 "deactivate", bookmarks_find_criterium2_deactivate_cb,
		 dialog);

	/* set data */
	dialog->controls = controls;
	
	/* this is done so that the enter key is not captured by the entry */
	gnome_dialog_editable_enters (GNOME_DIALOG (dialog->dialog),
				      GTK_EDITABLE (dialog->entry));

	/* parent dialog correctly */
	if (controls->type == STANDARD_BOOKMARKS_EDITOR)
		dialog_set_parent (dialog->dialog, controls->dialog);
	else
		dialog_set_parent (dialog->dialog, controls->window->wmain);

	/* when docked, make docked search entry unsensitive */
	/* if (controls->search_entry)
		gtk_widget_set_sensitive (controls->search_entry, FALSE); */

	/* update controls structure */
	controls->find_dialog = dialog;

	/* set default values */
	dialog->str = g_strdup ("");
	dialog->search_in = BOOKMARKS_SEARCH_IN_NAME;
	dialog->search_method = BOOKMARKS_SEARCH_METHOD_CONTAINS;

	gtk_object_unref (GTK_OBJECT (gxml));
}

void
bookmarks_find_prev_clicked_cb (GtkWidget *widget, BookmarksFindDialog *dialog)
{
	bookmarks_find_search (dialog, TRUE, FALSE);
}

void
bookmarks_find_next_clicked_cb (GtkWidget *widget, BookmarksFindDialog *dialog)
{
	bookmarks_find_search (dialog, FALSE, FALSE);
}

static gboolean
bookmarks_find_search (BookmarksFindDialog *dialog, gboolean prev,
		       gboolean wrap)
{
	gboolean found = FALSE;
	GList *backup = dialog->controls->search_position;
	gchar *utfstr = mozilla_locale_to_utf8(dialog->str);

	/* do the actual search */
	found = bookmarks_editor_search (dialog->controls, utfstr,
					 dialog->search_in,
					 dialog->search_method, prev, wrap);

	if (found == FALSE)
	{
		/* not found, wrap, and try again */
		found = bookmarks_editor_search (dialog->controls, utfstr,
					         dialog->search_in,
					         dialog->search_method,
						 prev, TRUE);
	}

	if (found == FALSE)
	{
		/* it's really not there; set search pos back to the previous
		 * search */
		dialog->controls->search_position = backup;
	}
	
	g_free (utfstr);

	return found;
}

void
bookmarks_find_cancel_clicked_cb (GtkWidget *widget,
				  BookmarksFindDialog *dialog)
{
	BookmarksEditorControls *controls = dialog->controls;
	gtk_widget_destroy (dialog->dialog);
	g_free (dialog->str);
	g_free (dialog);
	controls->find_dialog = NULL;
	
	/*if (controls->search_entry)
		gtk_widget_set_sensitive (controls->search_entry, TRUE);*/
}

void
bookmarks_find_entry_changed_cb (GtkEditable *editable,
				 BookmarksFindDialog *dialog)
{
	if (dialog->str) g_free (dialog->str);
	dialog->str = gtk_editable_get_chars (editable, 0, -1);
}

void
bookmarks_find_criterium1_deactivate_cb (GtkMenuShell *menushell,
					 BookmarksFindDialog *dialog)
{
	GtkWidget *active_item;

	/* find which item has been selected */
	active_item = gtk_menu_get_active (GTK_MENU (menushell));
	dialog->search_in = g_list_index (GTK_MENU_SHELL (menushell)->children,
					  active_item);
}

void
bookmarks_find_criterium2_deactivate_cb (GtkMenuShell *menushell,
					 BookmarksFindDialog *dialog)
{
	GtkWidget *active_item;

	/* find which item has been selected */
	active_item = gtk_menu_get_active (GTK_MENU (menushell));
	dialog->search_method =
		g_list_index (GTK_MENU_SHELL (menushell)->children,
			      active_item);
}

/* not used anymore, but kept in case we want to change the behaviour back */
#if 0
static void
bookmarks_find_update_prevnext_sensitivity (BookmarksFindDialog *dialog)
{
	gboolean prev_found, next_found;

	prev_found =
		bookmarks_find_can_search_prevnext (dialog, TRUE);
	gtk_widget_set_sensitive (dialog->prev_button, prev_found);
	next_found =
	 	bookmarks_find_can_search_prevnext (dialog, FALSE);
	gtk_widget_set_sensitive (dialog->next_button, next_found);
}

static gboolean
bookmarks_find_can_search_prevnext (BookmarksFindDialog *dialog,
				    gboolean backward)
{
	GtkCTree *ctree = GTK_CTREE (dialog->controls->ctree);
 	GList *l = backward ?
		g_list_previous (dialog->controls->search_position) :
		g_list_next (dialog->controls->search_position);

	if (dialog->controls->search_position == NULL)
		l = GTK_CLIST (dialog->controls->ctree)->row_list;
	
	for (; l; l = backward ? g_list_previous (l) : g_list_next (l))
	{
		BookmarkItem *b = gtk_ctree_node_get_row_data (ctree,
						GTK_CTREE_NODE (l));

		if (bookmarks_editor_search_matches (b, dialog->str,
		  				     dialog->search_in,
		   				     dialog->search_method))
		{
			return TRUE;
		}
	}

	return FALSE;
}
#endif
